# Mile High Maneuvers Romance System - Simplified
# Handles romance progression, scene routing, and relationship tracking

init python:
    # Heart level thresholds
    HEART_THRESHOLDS = {
        1: 0,    # Automatic after introduction
        2: 10,   # Need 10 points total
        3: 30,   # Need 30 points total
        4: 60,   # Need 60 points total
        5: 100   # Need 100 points total
    }
    
    # Points available per scene type (for reference)
    SCENE_POINTS = {
        "introduction": 5,
        "romance_1": 10,
        "romance_2": 20,
        "romance_3": 30,
        "romance_4": 40
    }
    
    # Catch-up point ranges by current heart level
    CATCHUP_POINTS = {
        0: (1, 2),  # 0→1 hearts
        1: (2, 3),  # 1→2 hearts
        2: (3, 4),  # 2→3 hearts
        3: (4, 5),  # 3→4 hearts
        4: (5, 6)   # 4→5 hearts
    }
    
    # NPC trait preferences - Standard format
    NPC_TRAIT_PREFERENCES = {
        "clutch": {"primary": "assertiveness", "secondary": "humor", "neutral": "flirtiness", "opposed": "submissiveness"},
        "lounge_grizz": {"primary": "flirtiness", "secondary": "assertiveness", "neutral": "humor", "opposed": "submissiveness"}
    }
    
    # NPCs with dual preferences (both traits give +2/-2)
    NPC_DUAL_PREFERENCES = {
        "marcus": {"dual_primary": ["assertiveness", "flirtiness"], "dual_opposed": ["humor", "submissiveness"]}
    }
    
    def get_heart_level(npc):
        """Get current heart level based on completed reward scenes"""
        base_name = npc.image_base
        
        # Start with 1 heart after introduction
        intro_scene = f"{base_name}_introduction"
        if has_completed_scene(npc, intro_scene):
            hearts = 1
        else:
            hearts = 0
        
        # Check each reward scene to award additional hearts
        for i in range(1, 5):  # Check rewards 1-4
            reward_scene = f"{base_name}_romance_reward_{i}"
            if has_completed_scene(npc, reward_scene):
                hearts = i + 1  # Reward 1 gives 2 hearts, reward 2 gives 3 hearts, etc.
            else:
                break  # Stop checking once we hit an uncompleted reward
        
        return min(hearts, 5)  # Cap at 5 hearts
    
    def get_potential_heart_level(npc):
        """Get potential heart level based on romance points (for progress tracking)"""
        if npc.romance >= 100: return 5
        elif npc.romance >= 60: return 4
        elif npc.romance >= 30: return 3
        elif npc.romance >= 10: return 2
        elif npc.romance >= 0: return 1
        else: return 0
    
    def get_heart_info(npc):
        """Get current hearts, points needed for next level, and threshold"""
        current_hearts = get_heart_level(npc)
        potential_hearts = get_potential_heart_level(npc)
        
        if current_hearts >= 5:
            return current_hearts, 0, None
        
        # Use potential hearts to determine next threshold
        next_threshold = HEART_THRESHOLDS[potential_hearts + 1] if potential_hearts < 5 else 100
        points_needed = max(0, next_threshold - npc.romance)
        return current_hearts, points_needed, next_threshold
    
    def calculate_trait_bonus(mc, npc, trait_used):
        """Calculate point modifier based on NPC's trait preferences"""
        if not trait_used:
            return 0
            
        # Check dual preferences first
        if npc.image_base in NPC_DUAL_PREFERENCES:
            prefs = NPC_DUAL_PREFERENCES[npc.image_base]
            if trait_used in prefs.get("dual_primary", []):
                return 2
            elif trait_used in prefs.get("dual_opposed", []):
                return -2
            return 0
        
        # Standard preferences
        prefs = NPC_TRAIT_PREFERENCES.get(npc.image_base, {})
        if trait_used == prefs.get("primary"): return 2
        elif trait_used == prefs.get("secondary"): return 1
        elif trait_used == prefs.get("neutral"): return 0
        elif trait_used == prefs.get("opposed"): return -1
        return 0
    
    def add_romance_points(npc, base_points=0, trait_used=None):
        """Add romance points based on trait used
        If trait_used is provided, points are determined ONLY by trait preference.
        Base_points are ignored when a trait is used."""
        if trait_used and mc:
            # When a trait is used, points are ONLY from trait preference
            points = calculate_trait_bonus(mc, npc, trait_used)
        else:
            # No trait used, use base_points
            points = base_points
        points = max(0, points)  # Never go negative
        npc.romance = min(100, npc.romance + points)  # Cap at 100
        return points
    
    def mark_scene_completed(npc, scene_name):
        """Mark a scene as completed"""
        if scene_name not in npc.scenes_completed:
            npc.scenes_completed.append(scene_name)
    
    def has_completed_scene(npc, scene_name):
        """Check if a scene has been completed"""
        return scene_name in npc.scenes_completed
    
    def get_available_catchup_scenes(npc):
        """Get catch-up scenes in order, then randomly"""
        base_name = npc.image_base
        unseen = []
        seen = []
        
        # Check first 10 catch-up scenes
        for i in range(1, 11):
            scene_name = f"{base_name}_catchup_{i}"
            # Trust that scenes exist if they follow naming convention
            if renpy.has_label(scene_name):
                if has_completed_scene(npc, scene_name):
                    seen.append(scene_name)
                else:
                    unseen.append(scene_name)
        
        # Return unseen in order, or random from seen if all completed
        if unseen:
            return unseen[0]  # First unseen scene
        elif seen:
            import random
            return random.choice(seen)  # Random from seen
        return None
    
    def get_next_romance_scene(npc):
        """Enforces strict linear progression: intro → romance_1 → reward_1 → romance_2 → reward_2, etc."""
        current_hearts = get_heart_level(npc)
        base_name = npc.image_base
        
        # 1. Check if introduction is needed
        intro_scene = f"{base_name}_introduction"
        if not has_completed_scene(npc, intro_scene):
            return "introduction", intro_scene
        
        # 2. Check what the next expected scene should be based on current hearts
        if current_hearts == 0:
            # Should be impossible if intro was seen, but just in case
            return None, None
        
        # Determine next main scene number (current_hearts corresponds to next romance scene)
        next_scene_num = current_hearts
        
        # 3. Check if we need to show the next main romance scene
        if next_scene_num <= 4:  # Max is romance_4
            romance_scene = f"{base_name}_romance_{next_scene_num}"
            if not has_completed_scene(npc, romance_scene):
                # Need to show this romance scene
                return "romance", romance_scene
            
            # 4. Romance scene was completed, check if reward scene is available
            potential_hearts = get_potential_heart_level(npc)
            if potential_hearts > current_hearts:
                # Player has enough points for next heart level
                reward_scene = f"{base_name}_romance_reward_{next_scene_num}"
                if not has_completed_scene(npc, reward_scene):
                    # Check if we need a bridge scene
                    bridge_scene = f"{base_name}_threshold_bridge_{next_scene_num}"
                    if not has_completed_scene(npc, bridge_scene) and renpy.has_label(bridge_scene):
                        return "bridge", bridge_scene
                    return "reward", reward_scene
        
        # 5. Not ready for next scene, offer catch-up scenes
        if current_hearts < 5:
            # Determine which threshold we're working towards
            next_threshold_hearts = min(current_hearts + 1, 5)
            if npc.romance < HEART_THRESHOLDS[next_threshold_hearts]:
                catchup = get_available_catchup_scenes(npc)
                if catchup:
                    return "catchup", catchup
        
        # No scenes available
        return None, None
    
    def set_npc_trait_preferences(npc, primary=None, secondary=None, neutral=None, opposed=None, dual_primary=None, dual_opposed=None):
        """Set trait preferences for an NPC"""
        if dual_primary and dual_opposed:
            NPC_DUAL_PREFERENCES[npc.image_base] = {
                "dual_primary": dual_primary,
                "dual_opposed": dual_opposed
            }
            NPC_TRAIT_PREFERENCES.pop(npc.image_base, None)
        else:
            NPC_TRAIT_PREFERENCES[npc.image_base] = {
                "primary": primary,
                "secondary": secondary,
                "neutral": neutral,
                "opposed": opposed
            }
            NPC_DUAL_PREFERENCES.pop(npc.image_base, None)

# Romance status display
screen romance_status(npc):
    frame:
        xalign 0.5
        yalign 0.1
        padding (20, 10)
        
        hbox:
            spacing 10
            text "[npc.character.name]" size 24
            
            # Heart display (earned hearts)
            $ hearts = get_heart_level(npc)
            $ potential_hearts = get_potential_heart_level(npc)
            for i in range(5):
                if i < hearts:
                    text "♥" color "#ff0066" size 30
                elif i < potential_hearts:
                    text "♥" color "#ff006660" size 30  # Semi-transparent for potential
                else:
                    text "♡" color "#666666" size 30
            
            if config.developer:
                text "([npc.romance]/100) H:[hearts]/[potential_hearts]" size 16 color "#888888"

# Romance notification
screen romance_notification(npc, message="Romance increased!"):
    timer 3.0 action Hide("romance_notification")
    
    frame:
        xalign 0.5
        yalign 0.9
        padding (30, 20)
        
        vbox:
            spacing 10
            xalign 0.5
            text message size 20 xalign 0.5
            
            hbox:
                spacing 5
                xalign 0.5
                $ hearts = get_heart_level(npc)
                for i in range(5):
                    if i < hearts:
                        text "♥" color "#ff0066" size 25
                    else:
                        text "♡" color "#666666" size 25

# Main romance scene router
label romance_scene_router(npc):
    $ scene_type, scene_label = get_next_romance_scene(npc)
    
    if scene_type is None:
        $ hearts = get_heart_level(npc)
        if hearts >= 5:
            "You've reached maximum romance level with [npc.character.name]!"
        else:
            "No romance scenes currently available with [npc.character.name]."
        return
    
    # Route to scene
    if scene_type == "introduction":
        call expression scene_label from _call_expression
    elif scene_type == "bridge":
        call expression scene_label from _call_expression_2
        # Bridge leads directly to reward
        $ reward_scene = scene_label.replace("threshold_bridge", "romance_reward")
        if renpy.has_label(reward_scene):
            call expression reward_scene from _call_expression_3
    else:
        call expression scene_label from _call_expression_4
    
    return

# Show romance progress
label show_romance_progress(npc):
    $ hearts, points_needed, next_threshold = get_heart_info(npc)
    $ potential_hearts = get_potential_heart_level(npc)
    
    if hearts < 5:
        if hearts < potential_hearts:
            "Current romance: [hearts] hearts earned, [potential_hearts] hearts potential ([npc.romance] points)"
            "Complete the next romance scene to earn your next heart!"
        else:
            "Current romance: [hearts] hearts ([npc.romance]/[next_threshold] points)"
            "Need [points_needed] more points to unlock the next scene."
    else:
        "Maximum romance reached: 5 hearts!"
    
    return

# Utility function for notifications
init python:
    def show_romance_notification(npc, message="Romance increased!"):
        renpy.show_screen("romance_notification", npc, message)
        # Force refresh of all screens to update heart display in say screen
        renpy.restart_interaction()


# Quick reference for developers:
# 1. Route to next scene: call romance_scene_router(npc)
# 2. Add points: $ points = add_romance_points(npc, trait_used="assertiveness")
#    - Single preference NPCs: primary=+2, secondary=+1, neutral=0, opposed=-1
#    - Dual preference NPCs: primary1/2=+2 each, opposed1/2=-2 each
#    - Base points are IGNORED when a trait is used
# 3. Mark complete: $ mark_scene_completed(npc, "scene_name")
# 4. Check hearts: $ hearts = get_heart_level(npc)  # Earned hearts (based on completed reward scenes)
# 5. Check potential: $ potential = get_potential_heart_level(npc)  # Based on points
# 6. Show notification: $ show_romance_notification(npc, "Message!")
#
# Scene progression is strictly linear:
# intro → romance_1 → reward_1 (2♥) → romance_2 → reward_2 (3♥) → etc.
# Hearts are ONLY awarded after completing reward scenes, not from points alone.

# Debug function for testing romance progression
label test_romance_progression(npc):
    "Testing romance progression for [npc.character.name]..."
    
    $ hearts = get_heart_level(npc)
    $ potential = get_potential_heart_level(npc)
    $ scene_type, scene_label = get_next_romance_scene(npc)
    
    "Current state:"
    "- Earned hearts: [hearts]"
    "- Potential hearts: [potential]"
    "- Romance points: [npc.romance]"
    "- Next scene type: [scene_type]"
    "- Next scene label: [scene_label]"
    
    "Completed scenes:"
    $ completed = ", ".join(npc.scenes_completed) if npc.scenes_completed else "None"
    "[completed]"
    
    return

