# Simple Mixology Minigame
init python:
    import random
    import time

    # Debug flag - set to True to show debugging information
    MIXOLOGY_DEBUG = True
    
    # Constants for recipe functionality
    RECIPE_CHANCE = 0.5  # Chance to spawn a needed ingredient
    
    # Game constants
    MAX_WRONG_CATCHES = 3  # Game ends after catching 3 wrong ingredients
    SPEED_INCREASE_PER_DRINK_FLAT = 100  # Flat 100 px/s speed increase per completed drink (doubled for difficulty)
    BASE_FALL_SPEED = 200  # Base pixels per second for falling ingredients
    TARGET_COMPLETED_ORDERS = 10  # Game also ends after successfully crafting 10 cocktails
    
    # Colors for UI elements
    UI_COLORS = {
        "caught": "#88ff88",    # Green for caught ingredients
        "next": "#ffff88",      # Yellow for next needed ingredient
        "remaining": "#888888", # Gray for remaining ingredients
        "recipe_name": "#ffdd88", # Gold for recipe name
        "message": "#ffffff",    # White for messages
        "miss_empty": "#444444", # Dark gray for empty miss slots
        "miss_filled": "#ff4444" # Red for filled miss slots
    }

    # Ingredient class to track falling ingredients
    class Ingredient:
        def __init__(self, name, image_path):
            self.name = name
            self.image_path = image_path
            self.x = 0
            self.y = 0
            self.width = 128  # Default width of ingredient images (updated from 64)
            self.height = 128  # Default height of ingredient images (updated from 64)
            self.speed = 0
            self.active = False
            self.caught = False
            self.missed = False
            self.debug_id = random.randint(1000, 9999)  # Unique ID for debugging
            
        def reset(self, screen_width, fall_speed):
            # Initialize at a random x position at the top of the screen, with 200px buffer on each side
            left_buffer = 200
            right_buffer = 200
            min_x = left_buffer + self.width // 2
            max_x = screen_width - right_buffer - self.width // 2
            
            # Ensure the spawn range is valid (min_x < max_x)
            if min_x >= max_x:
                # If the screen is too narrow, use center
                self.x = screen_width // 2
            else:
                self.x = random.randint(min_x, max_x)
                
            self.y = -self.height  # Start above the screen
            self.speed = fall_speed  # Use the provided fall speed (pixels per second)
            self.active = True
            self.caught = False
            self.missed = False
            if MIXOLOGY_DEBUG:
                print(f"Reset {self.name} (ID:{self.debug_id}) at x={self.x}, speed={self.speed:.1f} px/s")
            
        def update(self, dt):
            if not self.active:
                return
                
            # Update position based on time (pixels per second * seconds elapsed)
            self.y += self.speed * dt
            
        def is_caught(self, shaker_x, shaker_width, catch_zone, shaker_y):
            # If already caught or missed, return False
            if self.caught or self.missed or not self.active:
                return False
                
            # Get ingredient center and bottom coordinates
            ingredient_center_x = self.x + (self.width / 2)
            ingredient_bottom_y = self.y + self.height
            
            # Get catch zone boundaries
            catch_left = shaker_x - catch_zone
            catch_right = shaker_x + shaker_width + catch_zone
            
            # Check if ingredient bottom edge enters the top of the shaker
            return (ingredient_bottom_y >= shaker_y and
                    ingredient_bottom_y <= shaker_y + 20 and  # Allow small vertical tolerance
                    ingredient_center_x >= catch_left and
                    ingredient_center_x <= catch_right)
                    
        def is_offscreen(self, screen_height):
            # Check if the top of the ingredient has moved past the bottom of the screen
            # This ensures the entire ingredient is off-screen before being removed
            return self.y > screen_height
            
        def mark_caught(self):
            self.caught = True
            self.active = False  # Set active to False when caught
            if MIXOLOGY_DEBUG:
                print(f"Caught {self.name} (ID:{self.debug_id}) at y={self.y:.1f}")
            
        def mark_missed(self):
            self.missed = True
            self.active = False  # Set active to False when missed
            if MIXOLOGY_DEBUG:
                print(f"Missed {self.name} (ID:{self.debug_id}) at y={self.y:.1f}")
                
    # Recipe class to track drink recipes
    class DrinkRecipe:
        def __init__(self, name, ingredients, description="", difficulty=1):
            self.name = name
            self.ingredients = ingredients  # List of ingredient names
            self.description = description  # Optional flavor text
            self.difficulty = difficulty    # 1-3 scale of difficulty

        def get_ingredient_list_str(self):
            return ", ".join(self.ingredients)

    class MixologyGame:
        def __init__(self):
            self.screen_width = 1920
            self.screen_height = 1080
            self.shaker_width = 100
            self.shaker_height = 120
            self.shaker_x = (self.screen_width - self.shaker_width) // 2  # Centered for 1920 width
            self.shaker_y = self.screen_height - self.shaker_height - int(self.screen_height * 0.05)  # 5% above bottom for 1080 height
            self.shaker_speed = 100  # Pixels per keypress
            self.catch_zone = 40  # Extra pixels on each side where ingredients can be caught
            self.state = "playing"
            
            # Time tracking
            self.last_update_time = time.time()
            self.accumulated_time = 0.0  # For fixed timestep if needed
            self.frame_count = 0
            self.fps_update_time = time.time()
            self.current_fps = 0
            
            # Recipe-related properties
            self.current_recipe = None
            self.caught_ingredients = []  # List of ingredient names caught for current recipe
            self.orders_completed = 0
            self.combo_count = 0
            self.message = None
            self.message_timer = 0
            
            # Wrong catch tracking
            self.wrong_catches = 0  # Number of wrong ingredients caught
            
            # Speed management
            self.current_fall_speed = BASE_FALL_SPEED  # Current fall speed in pixels per second
            
            # Ingredient-related properties
            self.active_ingredients = []
            self.spawn_timer = 0
            self.spawn_interval = 1.5  # Seconds between ingredient spawns
            self.ingredients_caught = 0
            self.ingredients_missed = 0
            
            # Miles tracking (instead of score)
            self.base_miles = 50  # Everyone gets at least this many miles
            self.earned_miles = 0  # Additional miles earned through gameplay
            
            # Define available ingredients
            self.available_ingredients = [
                Ingredient("Tequila", "images/mixology/ingredient_tequila.png"),
                Ingredient("Gin", "images/mixology/ingredient_gin.png"),
                Ingredient("Vodka", "images/mixology/ingredient_vodka.png"),
                Ingredient("Rum", "images/mixology/ingredient_rum.png"),
                Ingredient("Tonic", "images/mixology/ingredient_tonic.png"),
                Ingredient("Soda", "images/mixology/ingredient_soda.png"),
                Ingredient("Orange Juice", "images/mixology/ingredient_orange_juice.png"),
                Ingredient("Cola", "images/mixology/ingredient_cola.png"),
                Ingredient("Cherry", "images/mixology/ingredient_cherry.png"),
                Ingredient("Lime", "images/mixology/ingredient_lime.png")
            ]
            
            # Define available recipes
            self.available_recipes = [
                # 2-Ingredient
                DrinkRecipe("Rum and Cola", ["Rum", "Cola"], "", 1),
                DrinkRecipe("Screwdriver", ["Vodka", "Orange Juice"], "", 1),
                DrinkRecipe("Gin and Juice", ["Gin", "Orange Juice"], "", 1),

                # 3-Ingredient
                DrinkRecipe("Gin and Tonic", ["Gin", "Tonic", "Lime"], "", 2),
                DrinkRecipe("Tequila Soda", ["Tequila", "Soda", "Lime"], "", 2),
                DrinkRecipe("Vodka Soda", ["Vodka", "Soda", "Lime"], "", 2),
                DrinkRecipe("Vodka Tonic", ["Vodka", "Tonic", "Lime"], "", 2),
                DrinkRecipe("Tequila Sunrise", ["Tequila", "Orange Juice", "Cherry"], "", 2),

                # 4-Ingredient
                DrinkRecipe("Knot Your Average Red-Eye", ["Rum", "Vodka", "Cola", "Cherry"], "", 3),
                DrinkRecipe("Tangy Tail-Wind", ["Gin", "Tonic", "Orange Juice", "Lime"], "", 3),

                # 5-Ingredient
                DrinkRecipe("Mile High Heat", ["Tequila", "Gin", "Vodka", "Rum", "Cola"], "", 3),
            ]
            
            # Select initial recipe
            self.select_new_recipe()
        
        def reset_timer(self):
            """Reset the timer - call this when the game screen is shown to prevent large initial dt"""
            self.last_update_time = time.time()
            self.fps_update_time = time.time()
            self.frame_count = 0
            if MIXOLOGY_DEBUG:
                print("Timer reset")
        
        def move_shaker(self, direction):
            if self.state != "playing":
                return
            new_x = self.shaker_x + direction * self.shaker_speed
            # Clamp to screen
            self.shaker_x = max(0, min(self.screen_width - self.shaker_width, new_x))
            if MIXOLOGY_DEBUG:
                print(f"Moved shaker to x={self.shaker_x}")
            
        def get_catch_zone_rect(self):
            # Returns the x, width of the catch zone
            catch_x = self.shaker_x - self.catch_zone
            catch_width = self.shaker_width + (self.catch_zone * 2)
            return catch_x, catch_width
        
        def set_message(self, text, duration=2.0):
            self.message = text
            self.message_timer = duration
            if MIXOLOGY_DEBUG:
                print(f"Message: {text} (Duration: {duration}s)")
                
        def select_new_recipe(self):
            if not self.available_recipes:
                return
                
            # Difficulty increases with orders completed
            max_difficulty = min(3, 1 + (self.orders_completed // 2))
            
            # Filter recipes by difficulty
            eligible_recipes = [r for r in self.available_recipes if r.difficulty <= max_difficulty]
            
            if not eligible_recipes:
                eligible_recipes = self.available_recipes
                
            # Choose a random recipe
            self.current_recipe = random.choice(eligible_recipes)
            self.caught_ingredients = []
            
            # Show recipe message
            self.set_message(f"New Order: {self.current_recipe.name}", 2.0)
            
        def process_catch(self, ingredient):
            if not self.current_recipe:
                # Simple miles if no recipe
                self.earned_miles += 5
                self.ingredients_caught += 1
                return
                
            # Check if this ingredient is needed for current recipe
            if ingredient.name in self.current_recipe.ingredients:
                # Check if we already caught this ingredient
                if ingredient.name not in self.caught_ingredients:
                    # Correct ingredient for the recipe (no sequence requirement)
                    self.caught_ingredients.append(ingredient.name)
                    
                    # Increase combo and miles
                    self.combo_count += 1
                    combo_bonus = min(3, 1 + (self.combo_count * 0.1))
                    miles_gained = int(5 * combo_bonus)  # Halved from 10 to 5
                    self.earned_miles += miles_gained
                    
                    # Check if recipe is complete (all ingredients caught)
                    if set(self.caught_ingredients) == set(self.current_recipe.ingredients):
                        self.complete_recipe()
                else:
                    # Already caught this ingredient - counts as wrong catch
                    self.wrong_catches += 1
                    self.earned_miles = max(0, self.earned_miles - 5)
                    self.set_message("Already caught this ingredient! -5 miles", 1.5)
                    self.combo_count = 0
                    
                    # Check if game over
                    if self.wrong_catches >= MAX_WRONG_CATCHES:
                        self.state = "results"
                        if MIXOLOGY_DEBUG:
                            print(f"Game over - {MAX_WRONG_CATCHES} wrong catches!")
            else:
                # Not needed for this recipe - wrong catch
                self.wrong_catches += 1
                self.earned_miles = max(0, self.earned_miles - 10)
                self.combo_count = 0
                self.set_message("Not needed! -10 miles", 1.5)
                
                # Check if game over
                if self.wrong_catches >= MAX_WRONG_CATCHES:
                    self.state = "results"
                    if MIXOLOGY_DEBUG:
                        print(f"Game over - {MAX_WRONG_CATCHES} wrong catches!")
                
            # Always increment caught counter
            self.ingredients_caught += 1
            
        def complete_recipe(self):
            # Award miles for completing the recipe (halved for balance)
            difficulty_bonus = self.current_recipe.difficulty * 12  # Halved from 25 to 12
            order_bonus = 15 + difficulty_bonus  # Base miles changed to 15
            self.earned_miles += order_bonus
            self.orders_completed += 1
            
            # Increase fall speed by 20%
            self.current_fall_speed += SPEED_INCREASE_PER_DRINK_FLAT
            
            # Check win condition: target number of completed orders
            if self.orders_completed >= TARGET_COMPLETED_ORDERS:
                # End the game on success
                self.set_message(
                    f"Order quota reached! {self.orders_completed}/{TARGET_COMPLETED_ORDERS} cocktails.",
                    3.0,
                )
                self.state = "results"
                if MIXOLOGY_DEBUG:
                    print(
                        f"Shift complete - target reached with {self.orders_completed} orders. Final speed: {self.current_fall_speed:.1f} px/s"
                    )
                return
            
            # Display success message
            self.set_message(
                f"Success! {self.current_recipe.name} completed! +{order_bonus} miles (Speed +{SPEED_INCREASE_PER_DRINK_FLAT} px/s)",
                3.0,
            )
            
            if MIXOLOGY_DEBUG:
                print(f"Recipe completed! New speed: {self.current_fall_speed:.1f} px/s")
            
            # Select a new recipe
            self.select_new_recipe()
            
        def update(self, dt_ignored):
            if self.state != "playing":
                return
            
            # Calculate real delta time
            current_time = time.time()
            dt = current_time - self.last_update_time
            self.last_update_time = current_time
            
            # Cap delta time to prevent large jumps (e.g., if game was paused)
            # Maximum of 1/30 second (33.3ms) per frame
            dt = min(dt, 0.0333)
            
            # Skip update if dt is too small (less than 1ms)
            if dt < 0.001:
                return
            
            # Update FPS counter
            self.frame_count += 1
            if current_time - self.fps_update_time >= 1.0:
                self.current_fps = self.frame_count
                self.frame_count = 0
                self.fps_update_time = current_time
            
            # Update message timer
            if self.message and self.message_timer > 0:
                self.message_timer -= dt
                if self.message_timer <= 0:
                    self.message = None
                
            # Update spawn timer
            self.spawn_timer -= dt
            if self.spawn_timer <= 0:
                self.spawn_ingredient()
                # Slightly reduce spawn interval as speed increases
                base_interval = 1.5
                min_interval = 0.8
                # For every 200 px/s increase, reduce interval by 20%
                speed_ratio = self.current_fall_speed / BASE_FALL_SPEED
                self.spawn_interval = max(min_interval, base_interval / speed_ratio)
                self.spawn_timer = self.spawn_interval
                
            # Update ingredient positions
            for ingredient in self.active_ingredients[:]:  # Use a copy to safely remove
                ingredient.update(dt)
                
                # Check if caught
                if ingredient.is_caught(self.shaker_x, self.shaker_width, self.catch_zone, self.shaker_y):
                    ingredient.mark_caught()
                    self.process_catch(ingredient)
                
                # Check if missed
                elif ingredient.is_offscreen(self.screen_height):
                    ingredient.mark_missed()
                    self.ingredients_missed += 1
                    
                # Remove if no longer active
                if not ingredient.active or ingredient.caught or ingredient.missed:
                    if MIXOLOGY_DEBUG:
                        print(f"Removing {ingredient.name} (ID:{ingredient.debug_id}) - state: active={ingredient.active}, caught={ingredient.caught}, missed={ingredient.missed}")
                    self.active_ingredients.remove(ingredient)
            
        def spawn_ingredient(self):
            chosen_template = None
            
            # Decide whether to spawn a needed ingredient or a distractor
            if self.current_recipe and random.random() < RECIPE_CHANCE:
                needed = [ing_name for ing_name in self.current_recipe.ingredients if ing_name not in self.caught_ingredients]
                if needed:
                    # Try to spawn the next needed ingredient in sequence
                    next_needed_name = needed[0]
                    chosen_template = next((ing for ing in self.available_ingredients
                                        if ing.name == next_needed_name), None)
            
            # If we didn't choose a needed one, pick a random one
            if not chosen_template:
                chosen_template = random.choice(self.available_ingredients)
                
            # Create a new instance
            ingredient = Ingredient(chosen_template.name, chosen_template.image_path)
            ingredient.reset(self.screen_width, self.current_fall_speed)
            
            # Add to active ingredients
            self.active_ingredients.append(ingredient)
            if MIXOLOGY_DEBUG:
                print(f"Spawned {ingredient.name} (ID:{ingredient.debug_id}) at x={ingredient.x}, speed={ingredient.speed:.1f} px/s")
                
        def get_results(self):
            # Total miles (base + earned)
            raw_miles = self.base_miles + self.earned_miles
            
            # Ensure minimum 50 miles
            raw_miles = max(50, raw_miles)
            
            # Return results
            return {
                "earned_miles": self.earned_miles,
                "base_miles": self.base_miles,
                "raw_miles": raw_miles,
                "caught": self.ingredients_caught,
                "missed": self.ingredients_missed,
                "completed": self.orders_completed,
                "wrong_catches": self.wrong_catches
            }

# The minigame screen
screen mixology_game(game):
    modal True
    zorder 200
    
    # Reset timer on screen show to prevent large initial dt
    on "show" action Function(game.reset_timer)
    
    # Timer for game updates - runs at roughly 60 FPS
    timer 0.016 repeat True action Function(game.update, 0.016)
    
    # Use the background image instead of solid black
    add "images/mixology/bg mixology_minigame.png" size (game.screen_width, game.screen_height)
    
    # Get catch zone rectangle dimensions
    $ catch_x, catch_width = game.get_catch_zone_rect()
    
    # Draw catch zone visualization (semi-transparent red rectangle)
    add Solid("#FF000080"):  # Red with 50% alpha
        xpos catch_x
        ypos game.shaker_y - 20  # Slightly above the shaker
        xsize catch_width
        ysize 20  # Height of the catch zone indicator
    
    # Current Recipe display
    if game.current_recipe and game.state == "playing":
        frame:
            xalign 0.5
            yalign 0.15
            padding (20, 15)
            background "#00000080"

            vbox:
                spacing 10
                xalign 0.5
                
                text "[game.current_recipe.name]" size 32 xalign 0.5 color UI_COLORS["recipe_name"] bold True
                
                # Show recipe description if available
                if game.current_recipe.description:
                    text "[game.current_recipe.description]" size 20 xalign 0.5 color "#FFFFFF" italic True
                
                null height 5  # Spacing
                
                hbox:
                    spacing 15
                    xalign 0.5

                    # Display ingredient slots
                    for i in range(len(game.current_recipe.ingredients)):
                        $ ing_name = game.current_recipe.ingredients[i]
                        $ is_caught = ing_name in game.caught_ingredients
                        
                        # Pre-calculate colors based on state
                        $ bg_color = "#333333"
                        $ border_color = UI_COLORS["caught"] if is_caught else "#666666"
                        
                        # Create border effect using nested frames
                        frame:
                            background border_color
                            padding (2, 2)
                            
                            frame:
                                xysize (116, 41)
                                background bg_color
                                padding (10, 5)

                                if is_caught:
                                    # Show name of caught ingredient with checkmark
                                    text ing_name + " ✓" color UI_COLORS["caught"] size 18 xalign 0.5 yalign 0.5
                                else:
                                    # Show uncaught ingredient
                                    text ing_name color UI_COLORS["remaining"] size 18 xalign 0.5 yalign 0.5
                
                null height 15  # Spacing before miss indicators
                
                # Miss indicators - 3 squares that fill with X's
                hbox:
                    spacing 10
                    xalign 0.5
                    
                    for i in range(MAX_WRONG_CATCHES):
                        $ miss_bg_color = UI_COLORS["miss_empty"] if i >= game.wrong_catches else UI_COLORS["miss_filled"]
                        frame:
                            xysize (50, 50)
                            background miss_bg_color
                            
                            if i < game.wrong_catches:
                                text "✗" size 30 xalign 0.5 yalign 0.5 color "#FFFFFF" bold True
    
    # Draw falling ingredients
    for ingredient in game.active_ingredients:
        if ingredient.active:
            # Try to load the image, use a placeholder if not found
            $ has_image = renpy.loadable(ingredient.image_path)
            if has_image:
                # Cast positions to int so Ren'Py treats them as absolute pixels
                $ draw_x = int(ingredient.x)
                $ draw_y = int(ingredient.y)
                add ingredient.image_path:
                    xpos draw_x
                    ypos draw_y
                    size (ingredient.width, ingredient.height)
            else:
                # Fallback - colored square with text
                $ draw_x = int(ingredient.x)
                $ draw_y = int(ingredient.y)
                add Solid("#FFFF00"):
                    xpos draw_x
                    ypos draw_y
                    xsize ingredient.width
                    ysize ingredient.height
                    
                text ingredient.name[0]:  # First letter of ingredient name
                    xpos draw_x + ingredient.width/2
                    ypos draw_y + ingredient.height/2
                    xanchor 0.5
                    yanchor 0.5
                    size 16
                    color "#000000"
            
            # Show debug ID when debug mode is enabled
            if MIXOLOGY_DEBUG:
                # Calculate text position above the ingredient
                $ text_y = max(0, draw_y - 30)  # Position above, but don't go off screen
                $ text_x = int(draw_x + ingredient.width/2)  # Center horizontally
                
                # Create a vbox to hold both background and text together
                fixed:
                    xpos draw_x - 25  # Position relative to ingredient
                    ypos text_y
                    xsize ingredient.width + 50
                    ysize 26
                    
                    # Background
                    add Solid("#000000AA")
                    
                    # Debug text with ingredient name and ID
                    text f"{ingredient.name} ({ingredient.debug_id})":
                        xalign 0.5  # Center within the fixed area
                        yalign 0.5  # Center vertically
                        size 16
                        color "#FFFF00"
                        outlines [(2, "#000000", 0, 0)]
    
    # Replace the red solid with the shaker image
    add "images/mixology/shaker.png":
        xpos game.shaker_x 
        ypos game.shaker_y
        size (game.shaker_width, game.shaker_height)
    
    # Display miles and stats
    vbox:
        xpos 20
        ypos 20
        spacing 10
        
        # On-screen exit for mobile users (mirrors Esc behavior)
        if game.state == "playing":
            frame:
                style "stats_button_frame"
                xalign 0.0
                ypos 0
                textbutton "X":
                    action Return(game.get_results())
                    style "top_bar_button"
                    text_style "top_bar_button_text"
                    xminimum 54
                    xmaximum 54
                    ysize 54
                    xalign 0.0
        
        text f"Miles: {game.base_miles + game.earned_miles}" size 24 color "#FFFFFF"
        
        hbox:
            spacing 15
            text f"Orders: {game.orders_completed}" size 20 color "#FFFFFF"
            
            # Show combo if active
            if game.combo_count > 1:
                text f"Combo: x{game.combo_count}" size 20 color "#FFDD66"
                
        text f"Caught: {game.ingredients_caught}" size 20 color "#FFFFFF"
        
        # Show current speed level
        if game.current_fall_speed > BASE_FALL_SPEED:
            text f"Speed: {int(game.current_fall_speed)} px/s (+{int(game.current_fall_speed - BASE_FALL_SPEED)})" size 20 color "#FFAA44"
    
    # Display feedback message if active
    # Commented out for now as it's getting in the way visually
    # if game.message:
    #     frame:
    #         xalign 0.5 
    #         ypos 250
    #         padding (20, 10)
    #         background "#00000080"
    #         
    #         text "[game.message]" size 28 color UI_COLORS["message"] xalign 0.5 bold True
    
    # Game over display
    if game.state == "results":
        frame:
            background "#00000080"
            xalign 0.5
            yalign 0.5
            padding (30, 30)
            
            vbox:
                spacing 20
                xalign 0.5
                
                text "Shift Complete!" size 40 xalign 0.5 color "#FFFFFF"
                if game.wrong_catches >= MAX_WRONG_CATCHES:
                    text f"Too many mistakes ({game.wrong_catches}/{MAX_WRONG_CATCHES})" size 24 xalign 0.5 color "#FF6666"
                elif game.orders_completed >= TARGET_COMPLETED_ORDERS:
                    text f"Drink Service Completed: ({game.orders_completed}/{TARGET_COMPLETED_ORDERS})" size 24 xalign 0.5 color "#66FF66"
                
                null height 10
                
                text f"Miles Earned: {game.earned_miles + game.base_miles}" size 30 xalign 0.5 color "#FFFFFF"
                text f"Orders Completed: {game.orders_completed}" size 24 xalign 0.5 color "#FFFFFF"
                text f"Ingredients Caught: {game.ingredients_caught}" size 24 xalign 0.5 color "#FFFFFF"
                
                # Show final speed level reached
                if game.current_fall_speed > BASE_FALL_SPEED:
                    text f"Final Speed: {int(game.current_fall_speed)} px/s (+{int(game.current_fall_speed - BASE_FALL_SPEED)})" size 20 xalign 0.5 color "#FFAA44"
                
                null height 20
                
                textbutton "Continue":
                    action Return(game.get_results())
                    xalign 0.5
    
    # Touch controls for mobile - only active during gameplay
    if game.state == "playing":
        # Define touch area boundaries
        $ touch_start_y = 300  # Start below UI elements
        $ touch_height = game.screen_height - touch_start_y
        $ left_width = game.screen_width // 2
        
        # Left touch area - move shaker left
        button:
            xpos 0
            ypos touch_start_y
            xsize left_width
            ysize touch_height
            idle_background None
            hover_background None
            action Function(game.move_shaker, -1)
            
        # Right touch area - move shaker right
        button:
            xpos left_width
            ypos touch_start_y
            xsize left_width
            ysize touch_height
            idle_background None
            hover_background None
            action Function(game.move_shaker, 1)
    
    # Controls
    key "K_LEFT" action Function(game.move_shaker, -1)
    key "K_RIGHT" action Function(game.move_shaker, 1)
    key "K_a" action Function(game.move_shaker, -1)
    key "K_d" action Function(game.move_shaker, 1)
    # Debug toggle with B key
    key "K_b" action [SetVariable("MIXOLOGY_DEBUG", not MIXOLOGY_DEBUG), Function(lambda: print(f"Debug mode: {MIXOLOGY_DEBUG}"))]
    
    # Allow escape to exit early
    key "K_ESCAPE" action Return(game.get_results())
    key "game_menu" action Return(game.get_results())
    
    # Show instructions
    frame:
        xalign 0.5
        yalign 0.05
        background None
        text "Use ←/→ or A/D to move. You can press escape to exit early." color "#fff" size 28
    
    # Display catch zone info for debugging (only when debug mode is on)
    if MIXOLOGY_DEBUG:
        text f"Catch Zone: {catch_x}-{catch_x + catch_width} (Width: {catch_width})":
            xpos 20
            ypos 100
            size 18
            color "#CCCCCC"
        
        # Display ingredient debug info
        vbox:
            xalign 1.0
            yalign 0.0
            xoffset -20
            yoffset 20
            
            text "Debug Info:" size 18 color "#FFFF00"
            text f"FPS: {game.current_fps}" size 16 color "#FFFFFF"
            text f"Active Ingredients: {len(game.active_ingredients)}" size 16 color "#FFFFFF"
            
            # Track first ingredient position
            if game.active_ingredients:
                $ first_ing = game.active_ingredients[0]
                text "--- First Ingredient ---" size 16 color "#FFFF00"
                text f"Name: {first_ing.name} (ID: {first_ing.debug_id})" size 16 color "#FFFFFF"
                text f"X Position: {first_ing.x:.1f}" size 16 color "#FFFFFF"
                text f"Y Position: {first_ing.y:.1f}" size 16 color "#FFFFFF"
                text f"Speed: {first_ing.speed:.1f} px/s" size 16 color "#FFFFFF"
                text f"Active: {first_ing.active}" size 16 color "#FFFFFF"
                text "---" size 16 color "#FFFF00"
            else:
                text "No active ingredients" size 16 color "#FFAA44"
                
            text f"Shaker Position: {game.shaker_x}" size 16 color "#FFFFFF"
            text f"Spawn Timer: {game.spawn_timer:.1f}s" size 16 color "#FFFFFF"
            text f"Spawn Interval: {game.spawn_interval:.1f}s" size 16 color "#FFFFFF"
            text f"Fall Speed: {game.current_fall_speed:.0f} px/s" size 16 color "#FFFFFF"
            text f"Wrong Catches: {game.wrong_catches}/{MAX_WRONG_CATCHES}" size 16 color "#FFFFFF"
            text f"Base Miles: {game.base_miles}" size 16 color "#FFFFFF"
            text f"Earned Miles: {game.earned_miles}" size 16 color "#FFFFFF"
            text f"Caught: {game.ingredients_caught}" size 16 color "#FFFFFF"
            text f"Missed: {game.ingredients_missed}" size 16 color "#FFFFFF"
            
            if game.current_recipe:
                text f"Current Recipe: {game.current_recipe.name}" size 16 color "#FFFFFF" 
                text f"Caught: {', '.join(game.caught_ingredients)}" size 16 color "#FFFFFF"

# Label to launch the minigame
label simple_mixology:
    $ mixology_game_instance = MixologyGame()
    call screen mixology_game(mixology_game_instance)
    $ results = _return
    
    # Get starting balance and raw miles earned
    $ starting_miles = mc.miles
    $ raw_miles = results["raw_miles"]
    
    # Pass the raw miles and skill name to the miles summary screen
    # The screen will handle looking up the skill value and calculating the multiplier
    call screen miles_summary_screen(starting_miles, raw_miles, "agility")
    
    # Calculate the miles to add to the player's account
    $ skill_multiplier = 1 + (mc.agility / 10.0)
    $ miles_earned = int(raw_miles * skill_multiplier)
    $ mc.miles += miles_earned
    
    # Extract results for dialogue
    $ earned_miles = results["earned_miles"]
    $ base_miles = results["base_miles"]
    $ completed = results["completed"]
    $ caught = results["caught"]
    
    # Display results
    "Mixology shift complete!"
    "Orders Completed: [completed]"
    "Ingredients Caught: [caught]"
    
    # Return the total miles earned (after multiplier) in a dictionary like at_your_service does
    $ return_dict = {"total_miles": miles_earned}
    return return_dict
