label regional_jet:
    # This file handles encounters on the regional jet, presenting a choice of NPCs to interact with
    
    scene bg regional_jet
    
    "The regional jet cabin is smaller and more intimate than the average aircraft. With fewer passengers and tighter quarters, interactions here tend to be more... personal."
    
    "As you make your way down the narrow aisle, checking seat belts and overhead bins, you can't help but notice how close you have to get to each passenger."
    
    "In such confined spaces, even casual contact becomes unavoidable."
    
    "The gentle hum of the engines provides a constant background noise, just loud enough to mask hushed conversations or... other sounds."
    
    # Refresh location NPC lists in case they changed earlier in the day
    $ Location.populate_all_locations()
    
    # Get the regional cabin location
    $ current_location = regional_cabin
    
    # Get all available NPCs from the regional cabin
    $ available_npcs = [npc for npc in current_location.NPCs if npc.regional_scene is not None]
    
    python:
        # Group NPCs by their scene
        scene_groups = {}
        for npc in available_npcs:
            scene_name = npc.regional_scene
            if scene_name not in scene_groups:
                scene_groups[scene_name] = []
            scene_groups[scene_name].append(npc)
        
        # Create scene entries with priority for unseen content
        scene_entries = []
        for scene_name, npcs in scene_groups.items():
            # Check if any NPC in this scene is unseen
            has_unseen = any(npc.enCounter == 0 for npc in npcs)
            # Get the minimum encounter count (prioritize less-seen scenes)
            min_encounters = min(npc.enCounter for npc in npcs)
            scene_entries.append({
                'scene_name': scene_name,
                'npcs': npcs,
                'has_unseen': has_unseen,
                'min_encounters': min_encounters
            })
        
        # Sort scene entries: unseen first, then by minimum encounters
        scene_entries.sort(key=lambda x: (not x['has_unseen'], x['min_encounters']))
        
        # Shuffle within priority groups to add randomness
        # Group entries by priority without using itertools
        priority_groups = {}
        for entry in scene_entries:
            key = (not entry['has_unseen'], entry['min_encounters'])
            if key not in priority_groups:
                priority_groups[key] = []
            priority_groups[key].append(entry)
        
        # Shuffle each group and rebuild the list
        shuffled_entries = []
        for key in sorted(priority_groups.keys()):
            group = priority_groups[key]
            random.shuffle(group)
            shuffled_entries.extend(group)
        
        # Take only the first few entries
        max_entries_to_show = 4
        final_entries = shuffled_entries[:max_entries_to_show]
    
    if len(final_entries) > 0:
        "As you scan the cabin, several passengers catch your attention. Who would you like to focus on?"
        
        python:
            # Build a menu with scene entries
            choices = []
            for entry in final_entries:
                npcs = entry['npcs']
                if len(npcs) == 1:
                    # Single NPC scene
                    npc = npcs[0]
                    choice_text = f"{npc.character.name} ({npc.species}) {npc.minigame_type}"
                else:
                    # Multi-NPC scene - show all names
                    names_and_species = [f"{npc.character.name} ({npc.species})" for npc in npcs]
                    # Use the first NPC's minigame type (they should all be the same for group scenes)
                    minigame_type = npcs[0].minigame_type if npcs[0].minigame_type else ""
                    choice_text = f"{' & '.join(names_and_species)} {minigame_type}"
                
                choices.append((choice_text, entry))
            
            # Provide an explicit sentinel value for the "focus on work" option so it
            # is always enabled and clickable on the custom choice screen.
            choices.append(("Focus on your work instead", "FOCUS_WORK"))
            
            # Display the menu and get the selected entry
            caption_text = "🍸 - Going Down Smooth (Agility Minigame)\n💬 - Cruising Altitude (Charisma Minigame)\n🛎️ - At Your Service (Intellect Minigame)"
            narrator(caption_text, interact=False)
            selected_entry = renpy.display_menu(choices)
        
        # Handle the player's selection. If they choose to focus on work, play a
        # short uneventful scene and end the flight. Otherwise, route to the
        # selected NPC scene.
        if selected_entry == "FOCUS_WORK":
            jump regional_jet_focus_on_work
        elif selected_entry is not None:
            $ selected_npcs = selected_entry['npcs']
            $ selected_scene = selected_entry['scene_name']
            
            # Build appropriate approach text
            if len(selected_npcs) == 1:
                "You decide to approach [selected_npcs[0].character.name]."
            else:
                $ npc_names = [npc.character.name for npc in selected_npcs]
                $ names_text = ", ".join(npc_names[:-1]) + " and " + npc_names[-1]
                "You decide to approach [names_text]."
            
            # Call the scene
            $ renpy.call(selected_scene)
            
            # Increment encounter count for all NPCs in the scene
            python:
                for npc in selected_npcs:
                    npc.encounter()
        else:
            # Fallback: treat any unexpected cancellation like choosing to focus
            # on work to keep the flow consistent.
            jump regional_jet_focus_on_work
    else:
        # Fallback if no valid passengers are found
        "The flight seems particularly uneventful today. None of the passengers seem interested in anything beyond the standard service."
        "You go through the motions of your job, somewhat disappointed by the lack of... excitement."
    
    return

label regional_jet_focus_on_work:
    "You keep your head down and work the aisle, pouring drinks and checking belts."
    "Small talk stays small. Call lights blink; you handle them one by one."
    "The engine hum settles into a steady backdrop as the flight moves along without incident."
    "By the time you stow the last cup and latch the cart, you're already on your final descent."
    return