# At Your Service V2 – Large-Grid Memory Minigame
# ----------------------------------------------
#  6 × 8 (48-card) grid version of the service memory game.
#  – Cards are shown for 5 s, then requests begin.
#  – Correct picks stay face-up; wrong picks re-hide after 1 s.
#  – Difficulty only changes the pool of item types that can appear.
#
#  Difficulty pools
#    easy   : 3 items  (Water, Headphones, Snack)
#    medium : 6 items  (+ Blanket, Seatbelt Extension, Pillow)
#    hard   : 9 items  (+ Lube, Magazine, Wing Pin)

init python:
    import random, math

    # Use same ITEM_IMAGES and ITEM_COLORS dicts already defined in the first
    # minigame. If this file is imported earlier, create fallbacks.
    if "ITEM_IMAGES" not in globals():
        ITEM_IMAGES = {}
    if "ITEM_COLORS" not in globals():
        ITEM_COLORS = {}

    # Define colors for each item type
    ITEM_COLORS = {
        "Water": "#3AAFFF",
        "Headphones": "#7D7DFF",
        "Snack": "#FFB347",
        "Blanket": "#FF6F91",
        "Seatbelt Extension": "#B19CD9",
        "Pillow": "#C6FFDD",
        "Lube": "#42F5B3",
        "Magazine": "#F5D142",
        "Wing Pin": "#FF914D",
    }

    # Populate ITEM_IMAGES with placeholder paths (these will be rendered as colored boxes with text)
    ITEM_IMAGES = {
        "Water": "placeholder_water",
        "Headphones": "placeholder_headphones",
        "Snack": "placeholder_snack",
        "Blanket": "placeholder_blanket",
        "Seatbelt Extension": "placeholder_seatbelt",
        "Pillow": "placeholder_pillow",
        "Lube": "placeholder_lube",
        "Magazine": "placeholder_magazine",
        "Wing Pin": "placeholder_wing_pin",
    }

    EASY_POOL   = ["Water", "Headphones", "Snack"]
    MED_POOL    = EASY_POOL + ["Blanket", "Seatbelt Extension", "Pillow"]
    HARD_POOL   = MED_POOL  + ["Lube", "Magazine", "Wing Pin"]

    # Maximum grid size (for hard) – used only for constants that need numbers
    MAX_GRID = 8  # 8×8

    # ------------------------------------------------------------------
    #  Helper – cached NPC sprite lookup (copied from v1) - REMOVED
    # ------------------------------------------------------------------
    # Removed get_npc_sprite_name function as we're not using NPCs anymore

    # ------------------------------------------------------------------
    #  Card class
    # ------------------------------------------------------------------
    class ItemCardV2:
        def __init__(self, item_name, row, col):
            self.item_name = item_name
            self.row = row
            self.col = col
            self.visible = True      # visible during initial reveal
            self.matched = False
            self.hide_timer = 0.0

    # ------------------------------------------------------------------
    #  Game logic class
    # ------------------------------------------------------------------
    class AtYourServiceGameV2:
        def __init__(self, intellect_value, difficulty="easy"):
            self.intellect   = intellect_value
            self.difficulty  = difficulty.lower()
            if self.difficulty not in ("easy", "medium", "hard"):
                self.difficulty = "easy"

            # Grid dimensions based on difficulty
            if self.difficulty == "easy":
                self.grid_rows = 4
                self.grid_cols = 4
            elif self.difficulty == "medium":
                self.grid_rows = 5
                self.grid_cols = 5
            else:  # hard
                self.grid_rows = 6
                self.grid_cols = 6

            self.total_cells = self.grid_rows * self.grid_cols

            # Timers
            self.total_time  = 60.0
            self.timer       = self.total_time
            self.reveal_time = 5.0
            self.reveal_phase = True
            self.reveal_timer = self.reveal_time

            # How long a wrongly-selected card stays visible (fixed 1 s)
            self.wrong_show_time = 1.0

            # Stats
            self.fulfilled_customers = 0
            self.wrong_clicks        = 0
            self.earned_miles        = 0

            # Build grid
            self.cards = []
            self._setup_cards()
            self.card_map = {(c.row, c.col): c for c in self.cards}

            # Current request - NO MORE NPC
            self.current_request = None

            self.state = "playing"

            self._next_request()   # Prepare first request after reveal

        # -------------------------- helpers ---------------------------
        def _item_pool(self):
            if self.difficulty == "easy":
                return EASY_POOL
            elif self.difficulty == "medium":
                return MED_POOL
            return HARD_POOL

        def _setup_cards(self):
            pool = self._item_pool()
            num_items_in_pool = len(pool)
            
            # Create a balanced distribution
            items_per_type = self.total_cells // num_items_in_pool
            remainder = self.total_cells % num_items_in_pool
            
            # Build the item list with as even distribution as possible
            balanced_items = []
            for i, item in enumerate(pool):
                # Add the base number for each item
                balanced_items.extend([item] * items_per_type)
                # Add one extra for some items if there's a remainder
                if i < remainder:
                    balanced_items.append(item)
            
            # Shuffle the items
            random.shuffle(balanced_items)
            
            # Assign to cards
            for idx in range(self.total_cells):
                item_name = balanced_items[idx]
                row = idx // self.grid_cols
                col = idx % self.grid_cols
                self.cards.append(ItemCardV2(item_name, row, col))

        def _next_request(self):
            # pick a random unmatched card
            choices = [c for c in self.cards if not c.matched]
            if choices:
                chosen = random.choice(choices)
                self.current_request = chosen.item_name
            else:
                self.current_request = None

            # No more NPC selection

        # --------------------------- update ---------------------------
        def update(self, dt):
            if self.state != "playing":
                return

            self.timer -= dt
            if self.timer <= 0:
                self.timer = 0
                self.state = "results"
                return

            if self.reveal_phase:
                self.reveal_timer -= dt
                if self.reveal_timer <= 0:
                    self.reveal_phase = False
                    for c in self.cards:
                        c.visible = False
                    # request already prep'd in __init__

            # handle hide timers
            for c in self.cards:
                if c.hide_timer > 0:
                    c.hide_timer -= dt
                    if c.hide_timer <= 0:
                        c.hide_timer = 0
                        if not c.matched:   # wrong card -> flip back
                            c.visible = False

        # ------------------------- input ------------------------------
        def select_card(self, card):
            if self.state != "playing" or self.reveal_phase or card.matched:
                return

            # show the card
            card.visible = True

            if card.item_name == self.current_request:
                card.matched = True
                self.fulfilled_customers += 1
                self.earned_miles += 10  # flat reward, tweak later
                # Check if all cards are now matched
                if all(c.matched for c in self.cards):
                    # Award completion bonus
                    self.earned_miles += 100
                    self.state = "results"
                    self.timer = 0
                else:
                    self._next_request()
            else:
                self.wrong_clicks += 1
                self.earned_miles = max(0, self.earned_miles - 5)
                card.hide_timer = self.wrong_show_time

        # ------------------------- results ----------------------------
        def get_results(self):
            base_miles = max(10, self.earned_miles)
            return {
                "fulfilled": self.fulfilled_customers,
                "wrong_clicks": self.wrong_clicks,
                "earned_miles": self.earned_miles,
                "raw_miles": base_miles,
            }

# --------------------------------------------------------------------
#  Screen
# --------------------------------------------------------------------
screen at_your_service_game_v2(game):
    modal True
    zorder 200

    timer 0.1 repeat True action Function(game.update, 0.1)

    hbox:
        spacing 0

        # Left – grid
        frame:
            xmaximum 1280
            yfill True
            background "#00000099"

            # top bar
            hbox:
                spacing 40
                xpos 20
                ypos 20
                text f"Time: {int(game.timer)}s" size 32 color "#ffffff"
                text f"Miles: {game.earned_miles}" size 32 color "#aaffaa"

            # dynamic grid
            $ rows = game.grid_rows
            $ cols = game.grid_cols
            grid rows cols:
                transpose False
                xalign 0.5
                yalign 0.5
                spacing 10
                for r in range(rows):
                    for c in range(cols):
                        $ card_obj = game.card_map[(r, c)]
                        $ is_shown = card_obj.visible or card_obj.matched
                        $ label_text = card_obj.item_name if is_shown else "?"
                        $ bg_color = ITEM_COLORS.get(card_obj.item_name, "#666666") if is_shown else "#333333"
                        $ txt_color = "#000000" if is_shown else "#FFFFFF"

                        button:
                            xminimum 128 yminimum 128
                            xmaximum 128 ymaximum 128
                            background bg_color
                            action Function(game.select_card, card_obj)
                            sensitive (not game.reveal_phase and not card_obj.matched)
                            text label_text size 16 color txt_color xalign 0.5 yalign 0.5 text_align 0.5 xmaximum 120 layout "subtitle"

        # Right – passenger panel
        frame:
            xfill True yfill True background "#111111AA"
            vbox:
                spacing 30 xalign 0.5 yalign 0.2
                text "Passenger Request" size 36 color "#FFD700" xalign 0.5
                if game.current_request:
                    text f"{game.current_request}" size 48 color "#FFFFFF" bold True xalign 0.5
                    
                    # Display large item placeholder instead of NPC sprite
                    frame:
                        xalign 0.5
                        ypos 200
                        xsize 300
                        ysize 300
                        background ITEM_COLORS.get(game.current_request, "#666666")
                        
                        # Add item name as text in the center
                        text game.current_request:
                            xalign 0.5
                            yalign 0.5
                            size 32
                            color "#000000"
                            text_align 0.5
                            xmaximum 280
                            layout "subtitle"
                            
                        # Add a subtle border
                        frame:
                            xfill True
                            yfill True
                            background None
                            foreground Frame(Solid("#FFFFFF33"), 5, 5)
                else:
                    text "Waiting..." size 32 color "#888888" italic True xalign 0.5

            if game.state == "results":
                frame:
                    background "#000000CC" xalign 0.5 yalign 0.5 padding (40,40)
                    vbox:
                        spacing 20
                        text "Service Complete!" size 48 color "#FFFFFF" xalign 0.5
                        text f"Passengers Served: {game.fulfilled_customers}" size 36 color "#FFFFFF" xalign 0.5
                        text f"Wrong Clicks: {game.wrong_clicks}" size 30 color "#FF8888" xalign 0.5
                        text f"Miles Earned: {game.earned_miles}" size 36 color "#aaffaa" xalign 0.5
                        textbutton "Continue" xalign 0.5 action Return(game.get_results())

    key "K_ESCAPE" action Return(game.get_results())

# --------------------------------------------------------------------
#  Label to launch V2
# --------------------------------------------------------------------
label at_your_service_minigame_v2:
    $ intellect_value = mc.intellect
    if intellect_value < 5:
        $ diff = "easy"
    elif intellect_value < 10:
        $ diff = "medium"
    else:
        $ diff = "hard"

    $ v2_game = AtYourServiceGameV2(intellect_value, diff)
    call screen at_your_service_game_v2(v2_game)
    $ res = _return

    # integrate miles similar to v1
    $ starting = mc.miles
    $ raw = res["raw_miles"]
    $ total_miles_earned = int(raw * (1 + mc.intellect/10.0))
    $ renpy.call_screen("miles_summary_screen", starting, raw, "intellect")
    $ mc.miles += total_miles_earned
    
    # Add the total miles to the result for scenario use
    $ res["total_miles"] = total_miles_earned

    "Service shift complete!"
    "Passengers Served: [res['fulfilled']]"
    "Wrong Clicks: [res['wrong_clicks']]"

    return res
