################################################################################
# CHARISMA MINIGAME V2 - FULLY MODULAR SYSTEM
################################################################################

init python:
    # Store variables to track the last dialogue for display during choices
    store.cruising_last_dialogue_speaker = None
    store.cruising_last_dialogue_text = ""
    store.cruising_last_dialogue_color = "#FFFFFF"  # Default white color
    
    # Maximum characters for a single line of narration (adjust as needed)
    CRUISING_MAX_TEXT_LENGTH = 200
    
    # Helper function to split long text into chunks
    def cruising_split_text(text, max_length=CRUISING_MAX_TEXT_LENGTH):
        """Split long text into multiple chunks at sentence or word boundaries"""
        if len(text) <= max_length:
            return [text]
        
        chunks = []
        sentences = text.split('. ')
        current_chunk = ""
        
        for i, sentence in enumerate(sentences):
            # Add period back if not the last sentence
            if i < len(sentences) - 1:
                sentence += '.'
            
            # If adding this sentence would exceed max length
            if current_chunk and len(current_chunk + ' ' + sentence) > max_length:
                # Save current chunk and start new one
                chunks.append(current_chunk.strip())
                current_chunk = sentence
            else:
                # Add to current chunk
                if current_chunk:
                    current_chunk += ' ' + sentence
                else:
                    current_chunk = sentence
            
            # If single sentence is too long, split by words
            if len(current_chunk) > max_length:
                words = current_chunk.split()
                temp_chunk = ""
                for word in words:
                    if temp_chunk and len(temp_chunk + ' ' + word) > max_length:
                        chunks.append(temp_chunk.strip())
                        temp_chunk = word
                    else:
                        if temp_chunk:
                            temp_chunk += ' ' + word
                        else:
                            temp_chunk = word
                current_chunk = temp_chunk
        
        # Add final chunk
        if current_chunk:
            chunks.append(current_chunk.strip())
        
        return chunks
    
    # Simple data classes for the minigame
    class CruisingDialogue:
        """Represents a single dialogue exchange in the minigame"""
        def __init__(self, speaker=None, text=""):
            self.speaker = speaker  # Character object or None for narration
            self.text = text
    
    class CruisingChoice:
        """Represents a single choice option"""
        def __init__(self, text, value, satisfaction_change=0, charisma_required=0, response=None, narration=None, fallback_response=None, fallback_narration=None, fallback_change=None):
            self.text = text
            self.value = value
            self.satisfaction_change = satisfaction_change
            self.charisma_required = charisma_required
            
            # Success responses (when charisma requirement is met)
            self.response = response or []
            self.narration = narration or []
            
            # Fallback responses (when charisma requirement is not met)
            self.fallback_response = fallback_response or []
            self.fallback_narration = fallback_narration or []
            self.fallback_change = fallback_change if fallback_change is not None else -10
    
    class CruisingInteraction:
        """Represents a single interaction with multiple choices"""
        def __init__(self, setup=None, question="What do you do?", choices=None):
            self.setup = setup or []  # List of CruisingDialogue objects for setup
            self.question = question
            self.choices = choices or []  # List of CruisingChoice objects
    
    class CruisingPassenger:
        """Represents a passenger for the charisma minigame"""
        def __init__(self, npc, intro=None, interactions=None, endings=None, background="bg regional_jet"):
            self.npc = npc
            self.intro = intro or []  # List of CruisingDialogue objects
            self.interactions = interactions or []  # List of CruisingInteraction objects
            self.endings = endings or {}  # Dict mapping outcome to label
            self.satisfaction = 50  # Starting satisfaction
            self.background = background  # Background image to use
            self.last_choice_value = None  # Track the last choice made
            
        def cruising_adjust_satisfaction(self, amount):
            """Adjust satisfaction and clamp to 0-100"""
            self.satisfaction += amount
            self.satisfaction = max(0, min(100, self.satisfaction))
            
        def cruising_get_outcome(self):
            """Determine outcome based on satisfaction"""
            # If the player explicitly chose to decline on the last choice, respect that
            if self.last_choice_value == "decline":
                return "decline"
                
            if self.satisfaction < 50:
                return "decline"
            elif self.satisfaction < 80:
                return "oral"
            else:
                return "anal"
                
        def cruising_get_ending_label(self, outcome):
            """Get the label for a specific outcome"""
            return self.endings.get(outcome, "")

    # Helper functions for creating dialogue
    def cruising_dialogue(speaker, text):
        """Create a CruisingDialogue with speaker"""
        return CruisingDialogue(speaker, text)
    
    def cruising_narration(text):
        """Create a CruisingDialogue for narration, automatically splitting long text"""
        # If text is already a list, process each item
        if isinstance(text, list):
            result = []
            for item in text:
                split_items = cruising_split_text(item)
                for split_item in split_items:
                    result.append(CruisingDialogue(None, split_item))
            return result
        else:
            # Single string - split if needed
            split_texts = cruising_split_text(text)
            if len(split_texts) == 1:
                return CruisingDialogue(None, text)
            else:
                return [CruisingDialogue(None, t) for t in split_texts]
    
    def cruising_choice(text, value, satisfaction=0, charisma=0, response=None, narration_text=None, fallback_response=None, fallback_narration=None, fallback_change=None):
        """Create a CruisingChoice with simplified parameters"""
        # Convert single strings to lists for consistency
        if isinstance(response, str):
            response = [response]
        if isinstance(narration_text, str):
            narration_text = cruising_split_text(narration_text)
        elif isinstance(narration_text, list):
            # Split each item in the list if needed
            split_narration = []
            for item in narration_text:
                split_narration.extend(cruising_split_text(item))
            narration_text = split_narration
        if isinstance(fallback_response, str):
            fallback_response = [fallback_response]
        if isinstance(fallback_narration, str):
            fallback_narration = cruising_split_text(fallback_narration)
        elif isinstance(fallback_narration, list):
            # Split each item in the list if needed
            split_fallback = []
            for item in fallback_narration:
                split_fallback.extend(cruising_split_text(item))
            fallback_narration = split_fallback
            
        return CruisingChoice(
            text=text,
            value=value,
            satisfaction_change=satisfaction,
            charisma_required=charisma,
            response=response,
            narration=narration_text,
            fallback_response=fallback_response,
            fallback_narration=fallback_narration,
            fallback_change=fallback_change
        )
    
    def cruising_interaction(setup=None, question="What do you do?", choices=None):
        """Create a CruisingInteraction"""
        return CruisingInteraction(setup, question, choices)
    
    # Helper function to track dialogue for display during choices
    def cruising_track_dialogue(speaker, text):
        """Track the last dialogue line for display during choice menus"""
        store.cruising_last_dialogue_speaker = speaker
        store.cruising_last_dialogue_text = text
        store.cruising_last_dialogue_color = cruising_get_speaker_color(speaker)
        # Debug output removed - no longer needed
            
    # Helper function to get the appropriate speaker for renpy.say
    def cruising_get_speaker_for_say(speaker):
        """Convert speaker to appropriate format for renpy.say()"""
        if speaker is None:
            return None
        elif hasattr(speaker, '__class__') and speaker.__class__.__name__ == 'Player':
            print "Player Object"
            # It's a Player object, always get the current mc.name
            # instead of the stored name from when the dialogue was created
            return store.mc.name
        elif hasattr(speaker, 'character'):
            # It's an NPC object, return the character
            return speaker.character
        elif hasattr(speaker, '__call__'):
            # It's likely a Ren'Py Character object (which is callable)
            # Return it directly
            return speaker
        else:
            # Fallback - try to use it directly
            print "Fallback"
            return speaker
            
    # Helper function to get speaker color
    def cruising_get_speaker_color(speaker):
        """Extract the color from a speaker object"""
        if speaker is None:
            return "#8899AA"  # Subtle blue-gray for narrator (from screens.rpy)
        elif hasattr(speaker, '__class__') and speaker.__class__.__name__ == 'Player':
            # It's a Player object, get color from the current mc
            return store.mc.color if hasattr(store.mc, 'color') else "#000000"
        elif hasattr(speaker, 'character'):
            # It's an NPC object - use manual color mapping
            
            # Manual mapping for all known characters
            character_colors = {
                "Jin": "#F4A460",           # Sandy brown for tiger
                "Chop": "#FFB6C1",          # Light pink for pig
                "Finley": "#8B5A2B",        # Brown for deer
                "Boris": "#8B4513",         # Brown for bear
                "Grizz": "#704214",         # Dark brown for bear
                "Fang": "#4F4F4F",          # Dark gray for wolf
                "Brutus": "#8C8C8C",        # Gray for rhino
                "Wallow": "#4a5253",        # Cadet blue for hippo
                "Dash": "#F0C05A",          # Golden yellow for cheetah
                "Professor Ivory": "#E8E4C9", # Ivory color for elephant
                "Garrick": "#B87333",       # Copper brown for warthog
                "Chuckles": "#62dddf",      # Mohawk color for hyena
                "Quincy": "#C0C0C0",        # Silver gray for mouse/rat
                "Douglas": "#8B7355",       # Wood brown for moose
                "Balthazar": "#0A0A0A",     # Deep black for black bull
                "Zephyr": "#c64f5d",        # Pink for warthog
                "Finn": "#D2B48C",          # Tan for meerkat
                "Kogo": "#4C4C4C",          # Dark gray for gorilla
                "Brodie": "#D4AF37",        # Gold for stallion/horse
                "Black Rhino": "#464646",   # Dark gray for black rhino
                "Wesley": "#8B4513",         # Brown color for minotaur
                "Captain Sterling": "#808080", # Gray for the older wolf
                "Trent": "#FFD700",         # Gold for the younger wolf's golden eyes
                "Thick Warthog": "#B22222", # Dark red for warthog
                "Passive Rabbit": "#E6E6FA", # Lavender for rabbit
                "Penwick": "#2E2E2E",       # Dark gray for panda
                "Marcus": "#7C7C7C",        # Silver-gray for wolf
                "Bradley": "#8B4513",       # Brown for bear
                "Nate": "#4B0000",          # Dark red/maroon for horse
            }
            
            # Get character name
            char_name = None
            if hasattr(speaker.character, 'name'):
                char_name = speaker.character.name
                
            if char_name and char_name in character_colors:
                return character_colors[char_name]
                
            # Fallback to black
            return "#000000"
        elif hasattr(speaker, 'name'):
            # It's likely a Ren'Py Character object passed directly
            # Use the same manual mapping
            character_colors = {
                "Jin": "#F4A460",           # Sandy brown for tiger
                "Chop": "#FFB6C1",          # Light pink for pig
                "Finley": "#8B5A2B",        # Brown for deer
                "Boris": "#8B4513",         # Brown for bear
                "Grizz": "#704214",         # Dark brown for bear
                "Fang": "#4F4F4F",          # Dark gray for wolf
                "Brutus": "#8C8C8C",        # Gray for rhino
                "Wallow": "#4a5253",        # Cadet blue for hippo
                "Dash": "#F0C05A",          # Golden yellow for cheetah
                "Professor Ivory": "#E8E4C9", # Ivory color for elephant
                "Garrick": "#B87333",       # Copper brown for warthog
                "Chuckles": "#62dddf",      # Mohawk color for hyena
                "Quincy": "#C0C0C0",        # Silver gray for mouse/rat
                "Douglas": "#8B7355",       # Wood brown for moose
                "Balthazar": "#0A0A0A",     # Deep black for black bull
                "Zephyr": "#c64f5d",        # Pink for warthog
                "Finn": "#D2B48C",          # Tan for meerkat
                "Kogo": "#4C4C4C",          # Dark gray for gorilla
                "Brodie": "#D4AF37",        # Gold for stallion/horse
                "Black Rhino": "#464646",   # Dark gray for black rhino
                "Wesley": "#8B4513",         # Brown color for minotaur
                "Captain Sterling": "#808080", # Gray for the older wolf
                "Trent": "#FFD700",         # Gold for the younger wolf's golden eyes
                "Thick Warthog": "#B22222", # Dark red for warthog
                "Passive Rabbit": "#E6E6FA", # Lavender for rabbit
                "Penwick": "#2E2E2E",       # Dark gray for panda
                "Marcus": "#7C7C7C",        # Silver-gray for wolf
                "Bradley": "#8B4513",       # Brown for bear
                "Nate": "#4B0000",          # Dark red/maroon for horse
            }
            
            if speaker.name in character_colors:
                return character_colors[speaker.name]
                
            # Fallback to black
            return "#000000"
        else:
            # Fallback to black
            return "#000000"

# Enhanced UI styles with text wrapping support
style cruising_text:
    color "#FFFFFF"
    size 22
    text_align 0.0
    layout "subtitle"  # Enables text wrapping
    
style cruising_dialogue_text:
    color "#FFFFFF"
    size 20
    text_align 0.0
    layout "subtitle"
    
style cruising_choice_text:
    color "#FFFFFF"
    size 18
    text_align 0.0
    layout "subtitle"
    
style cruising_question_text:
    color "#FFFFFF"
    size 24
    text_align 0.5
    layout "subtitle"
    bold True

# Main minigame screen with improved text wrapping
screen cruising_minigame_v2(passenger):
    modal True
    
    # Background
    add passenger.background
    
    # Character display
    if renpy.has_image(passenger.npc.image_base):
        add passenger.npc.image_base at Transform(zoom=0.8, xpos=0.75, ypos=0.6, xanchor=0.5, yanchor=0.5)
    
    # Satisfaction meter
    frame:
        xalign 0.95
        yalign 0.1
        xsize 150
        ysize 300
        background "#222222AA"
        padding (15, 15)
        
        vbox:
            spacing 10
            xalign 0.5
            text "Satisfaction" style "cruising_text" color "#AAFFAA" font "gui/CreteRound-Regular.ttf"
            bar value passenger.satisfaction range 100:
                xsize 120
                ysize 250
                bar_vertical True
                top_bar "#FFFFFF"
                bottom_bar "#AAFFAA"
                thumb None
    
    # Mood indicator
    $ mood = cruising_get_passenger_mood(passenger.satisfaction)
    $ mood_color = cruising_get_mood_color(mood)
    frame:
        xalign 0.75
        yalign 0.35
        xsize 200
        ysize 40
        background "#222222AA"
        padding (10, 5)
        
        text mood:
            xalign 0.5
            yalign 0.5
            color mood_color
            size 24
            bold True
            font "gui/CreteRound-Regular.ttf"
    
    # Charisma display
    frame:
        xalign 0.95
        yalign 0.5
        xsize 150
        ysize 80
        background "#222222AA"
        padding (15, 15)
        
        text "Charisma: [mc.charisma]" style "cruising_text" font "gui/CreteRound-Regular.ttf" xalign 0.5

# Choice menu screen with proper text wrapping and previous dialogue display
screen cruising_choice_menu_v2(passenger, question, choices):
    modal True
    
    # Hide the standard dialogue window
    $ renpy.hide_screen("say")
    
    # Background
    add passenger.background
    
    # Character display
    if renpy.has_image(passenger.npc.image_base):
        add passenger.npc.image_base at Transform(zoom=0.8, xpos=0.75, ypos=0.6, xanchor=0.5, yanchor=0.5)
    
    # Satisfaction meter (same as main screen)
    frame:
        xalign 0.95
        yalign 0.1
        xsize 150
        ysize 300
        background "#222222AA"
        padding (15, 15)
        
        vbox:
            spacing 10
            xalign 0.5
            text "Satisfaction" style "cruising_text" color "#AAFFAA" font "gui/CreteRound-Regular.ttf"
            bar value passenger.satisfaction range 100:
                xsize 120
                ysize 250
                bar_vertical True
                top_bar "#FFFFFF"
                bottom_bar "#AAFFAA"
                thumb None
    
    # Previous dialogue display - styled identical to say screen
    if store.cruising_last_dialogue_text:
        window:
            xalign 0.5
            #yalign gui.textbox_yalign
            ypos 0.95  # Position higher than normal dialogue (0.95) to avoid choice conflicts
            ysize gui.textbox_height
            xsize 1200
            background "gui/boardingpass_textbox.svg"
            
            # "NAME:" label
            text "NAME:" style "name_label"
            
            # Character name or "Narrator"
            if store.cruising_last_dialogue_speaker:
                # Get the actual name based on speaker type
                python:
                    if hasattr(store.cruising_last_dialogue_speaker, '__class__') and store.cruising_last_dialogue_speaker.__class__.__name__ == 'Player':
                        speaker_display_name = store.mc.name
                    elif hasattr(store.cruising_last_dialogue_speaker, 'character') and hasattr(store.cruising_last_dialogue_speaker.character, 'name'):
                        speaker_display_name = store.cruising_last_dialogue_speaker.character.name
                    elif hasattr(store.cruising_last_dialogue_speaker, 'name'):
                        # It's likely a Ren'Py Character object
                        speaker_display_name = store.cruising_last_dialogue_speaker.name
                    else:
                        speaker_display_name = "Unknown"
                
                text speaker_display_name:
                    style "say_label"
                    color store.cruising_last_dialogue_color
            else:
                text "Narrator" style "narrator_name"
            
            # MHM text in corner
            text "MHM" style "mhm_label"
            
            # Dialogue text with conditional italic
            if store.cruising_last_dialogue_speaker and store.cruising_last_dialogue_speaker.name not in ["Narrator", "Tutorial"]:
                text store.cruising_last_dialogue_text style "say_dialogue" italic True
            else:
                text store.cruising_last_dialogue_text style "say_dialogue"
    
    # Question frame with text wrapping
    frame:
        xalign 0.2
        yalign 0.15
        xsize 500
        background "#222222AA"
        padding (20, 15)
        
        text question:
            style "cruising_question_text"
            font "gui/CreteRound-Regular.ttf"
            xalign 0.5
            xfill True
    
    # Choices frame with text wrapping
    frame:
        xalign 0.2
        yalign 0.5
        xsize 500
        background "#222222AA"
        padding (20, 20)
        
        vbox:
            spacing 15
            xfill True
            
            for choice_obj in choices:
                $ can_choose = choice_obj.charisma_required <= mc.charisma
                $ display_text = choice_obj.text
                
                if choice_obj.charisma_required > 0:
                    if can_choose:
                        $ display_text += " (Charisma [choice_obj.charisma_required])"
                    else:
                        $ display_text += " (Requires Charisma [choice_obj.charisma_required])"
                
                if can_choose:
                    textbutton display_text:
                        action Return(choice_obj)
                        xfill True
                        background "#444444"
                        hover_background "#666666"
                        padding (15, 10)
                        text_style "cruising_choice_text"
                        text_font "gui/CreteRound-Regular.ttf"
                        text_color "#FFFFFF"
                        text_hover_color "#AAFFAA"
                        text_xalign 0.0
                else:
                    frame:
                        background "#333333"
                        xfill True
                        padding (15, 10)
                        
                        text display_text:
                            style "cruising_choice_text"
                            font "gui/CreteRound-Regular.ttf"
                            color "#888888"
                            xalign 0.0
                            xfill True

# Helper functions
init python:
    def cruising_get_passenger_mood(satisfaction):
        """Return mood description based on satisfaction"""
        if satisfaction < 20:
            return "Uncomfortable"
        elif satisfaction < 40:
            return "Disinterested"
        elif satisfaction < 60:
            return "Interested"
        elif satisfaction < 80:
            return "Eager"
        else:
            return "Aroused"
            
    def cruising_get_mood_color(mood):
        """Return color for mood"""
        colors = {
            "Uncomfortable": "#FF3333",
            "Disinterested": "#FF9933",
            "Interested": "#FFFF33",
            "Eager": "#99FF33",
            "Aroused": "#33FF33"
        }
        return colors.get(mood, "#FFFFFF")

# Main minigame label
label cruising_minigame_v2(passenger_data):
    $ renpy.block_rollback()
    
    # Create passenger object
    $ passenger = passenger_data
    
    # Show introduction
    if passenger.intro:
        call cruising_show_dialogue(passenger.intro) from _call_cruising_show_dialogue
    
    # Process each interaction
    $ interaction_index = 0
    while interaction_index < len(passenger.interactions):
        $ current_interaction = passenger.interactions[interaction_index]
        
        # Show setup dialogue if any
        if current_interaction.setup:
            call cruising_show_dialogue(current_interaction.setup) from _call_cruising_show_dialogue_1
        
        # Hide the dialogue window before showing choices
        window hide
        
        # Show choice menu
        $ chosen_choice = renpy.call_screen("cruising_choice_menu_v2", 
                                            passenger=passenger, 
                                            question=current_interaction.question, 
                                            choices=current_interaction.choices)
        
        # Process the choice
        call cruising_process_choice(passenger, chosen_choice) from _call_cruising_process_choice
        
        $ interaction_index += 1
    
    # Determine outcome
    $ outcome = passenger.cruising_get_outcome()
    $ ending_label = passenger.cruising_get_ending_label(outcome)
    
    # Jump to ending if available
    if ending_label:
        call expression ending_label from _call_expression_1
    
    return outcome

# Helper label to show dialogue sequences
label cruising_show_dialogue(dialogue_list):
    python:
        # Flatten the list in case cruising_narration returned lists
        flattened_list = []
        for item in dialogue_list:
            if isinstance(item, list):
                flattened_list.extend(item)
            else:
                flattened_list.append(item)
        
        for dialogue_item in flattened_list:
            if isinstance(dialogue_item, CruisingDialogue):
                if dialogue_item.speaker:
                    # Track this dialogue for display during choices
                    cruising_track_dialogue(dialogue_item.speaker, dialogue_item.text)
                    # Use helper function to get appropriate speaker format
                    speaker_for_say = cruising_get_speaker_for_say(dialogue_item.speaker)
                    renpy.say(speaker_for_say, dialogue_item.text)
                else:
                    # Track this narration for display during choices
                    cruising_track_dialogue(None, dialogue_item.text)
                    renpy.say(None, dialogue_item.text)
            elif isinstance(dialogue_item, str):
                # Support for simple strings as narration
                cruising_track_dialogue(None, dialogue_item)
                renpy.say(None, dialogue_item)
    return

# Helper label to process a choice
label cruising_process_choice(passenger, choice):
    # Check charisma requirement
    if choice.charisma_required > mc.charisma:
        # Use fallback
        $ satisfaction_change = choice.fallback_change
        $ response_list = choice.fallback_response
        $ narration_list = choice.fallback_narration
    else:
        # Use normal response
        $ satisfaction_change = choice.satisfaction_change
        $ response_list = choice.response
        $ narration_list = choice.narration
    
    # Show NPC response
    if response_list:
        python:
            for response_text in response_list:
                cruising_track_dialogue(passenger.npc, response_text)
                # Use helper function to get appropriate speaker format
                speaker_for_say = cruising_get_speaker_for_say(passenger.npc)
                renpy.say(speaker_for_say, response_text)
    
    # Show narration
    if narration_list:
        python:
            for narration_text in narration_list:
                cruising_track_dialogue(None, narration_text)
                renpy.say(None, narration_text)
    
    # Update satisfaction
    $ passenger.cruising_adjust_satisfaction(satisfaction_change)
    
    # Show satisfaction change notification
    if satisfaction_change > 0:
        $ renpy.notify("+[satisfaction_change] Satisfaction")
    elif satisfaction_change < 0:
        $ renpy.notify("[satisfaction_change] Satisfaction")
    
    # Update last choice value
    $ passenger.last_choice_value = choice.value
    
    return

# Entry point that maintains compatibility with existing system
label cruising_minigame_2_v2(passenger_name):
    $ renpy.block_rollback()
    
    # Find the NPC
    python:
        passenger_npc = None
        for npc in NPC.get_all_npcs():
            if npc.character.name.lower() == passenger_name.lower():
                passenger_npc = npc
                break
    
    if passenger_npc is None:
        return "decline"
    
    # Check if NPC has v2 data
    if hasattr(passenger_npc, 'cruising_v2_data'):
        $ outcome = renpy.call_in_new_context("cruising_minigame_v2", passenger_npc.cruising_v2_data)
        return outcome
    

    # Fall back to original system
    $ outcome = renpy.call_in_new_context("cruising_minigame_2", passenger_name)
    return outcome