﻿################################################################################
## Initialization
################################################################################

init offset = -1

# Custom action to properly refresh the player stats screen
init python:
    import time
    
    class RefreshPlayerStats(Action):
        def __call__(self):
            # Hide any existing stats screen completely
            renpy.hide_screen("player_stats", immediately=True)
            
            # Force all screens to update
            renpy.restart_interaction()
            
            # Show the screen with a unique timestamp to force complete re-evaluation
            renpy.show_screen("player_stats", refresh_time=time.time())
            
            # Return None to prevent dialogue advancement
            return None
    
    # Custom action for loading that stops main menu music
    class LoadWithMusicStop(Action):
        def __init__(self, slot):
            self.slot = slot
            
        def __call__(self):
            # Stop the main menu music with fadeout
            renpy.music.stop(channel='music', fadeout=2.0)
            # Then load the file
            return FileAction(self.slot).__call__()


################################################################################
## Styles
################################################################################

style default:
    properties gui.text_properties()
    language gui.language

style input:
    properties gui.text_properties("input", accent=True)
    adjust_spacing False
    color "#000000"  # Set text color to black

style hyperlink_text:
    properties gui.text_properties("hyperlink", accent=True)
    hover_underline True

style gui_text:
    properties gui.text_properties("interface")


style button:
    properties gui.button_properties("button")

style button_text is gui_text:
    properties gui.text_properties("button")
    yalign 0.5


style label_text is gui_text:
    properties gui.text_properties("label", accent=True)

style prompt_text is gui_text:
    properties gui.text_properties("prompt")


style bar:
    ysize gui.bar_size
    left_bar Frame("gui/bar/left.png", gui.bar_borders, tile=gui.bar_tile)
    right_bar Frame("gui/bar/right.png", gui.bar_borders, tile=gui.bar_tile)

style vbar:
    xsize gui.bar_size
    top_bar Frame("gui/bar/top.png", gui.vbar_borders, tile=gui.bar_tile)
    bottom_bar Frame("gui/bar/bottom.png", gui.vbar_borders, tile=gui.bar_tile)

style scrollbar:
    ysize gui.scrollbar_size
    base_bar Frame("gui/scrollbar/horizontal_[prefix_]bar.png", gui.scrollbar_borders, tile=gui.scrollbar_tile)
    thumb Frame("gui/scrollbar/horizontal_[prefix_]thumb.png", gui.scrollbar_borders, tile=gui.scrollbar_tile)

style vscrollbar:
    xsize gui.scrollbar_size
    base_bar Frame("gui/scrollbar/vertical_[prefix_]bar.png", gui.vscrollbar_borders, tile=gui.scrollbar_tile)
    thumb Frame("gui/scrollbar/vertical_[prefix_]thumb.png", gui.vscrollbar_borders, tile=gui.scrollbar_tile)

style slider:
    ysize gui.slider_size
    base_bar Frame("gui/slider/horizontal_[prefix_]bar.png", gui.slider_borders, tile=gui.slider_tile)
    thumb "gui/slider/horizontal_[prefix_]thumb.png"

style vslider:
    xsize gui.slider_size
    base_bar Frame("gui/slider/vertical_[prefix_]bar.png", gui.vslider_borders, tile=gui.slider_tile)
    thumb "gui/slider/vertical_[prefix_]thumb.png"


style frame:
    padding gui.frame_borders.padding
    background Frame("gui/frame.png", gui.frame_borders, tile=gui.frame_tile)



################################################################################
## In-game screens
################################################################################


## Say screen ##################################################################
##
## The say screen is used to display dialogue to the player. It takes two
## parameters, who and what, which are the name of the speaking character and
## the text to be displayed, respectively. (The who parameter can be None if no
## name is given.)
##
## This screen must create a text displayable with id "what", as Ren'Py uses
## this to manage text display. It can also create displayables with id "who"
## and id "window" to apply style properties.
##
## https://www.renpy.org/doc/html/screen_special.html#say

screen say(who, what):
    style_prefix "say"

    window:
        id "window"

        if who is not None:
            text "NAME:" style "name_label"
            # Check if this character is romanceable and modify the displayed name
            if is_romanceable_character(who):
                $ npc = NPC.get_npc_by_character_name(who)
                $ max_hearts = 5
                $ current_hearts = 0 if npc is None else max(0, min(getattr(npc, "hearts", 0), max_hearts))
                $ hearts_display = "♥" * current_hearts + "♡" * (max_hearts - current_hearts)
                text who + " - {color=#AAAAAA}" + hearts_display + "{/color}" id "who"
            else:
                text who id "who"
        else:
            # Show "Narrator" for lines with no speaker
            text "NAME:" style "name_label"
            text "Narrator" style "narrator_name"

        # Add MHM text to the corner circle
        text "MHM" style "mhm_label"

        # Make text italic for characters other than Narrator and Tutorial
        if who is not None and who != "Narrator" and who != "Tutorial":
            text what id "what" italic True
        else:
            text what id "what"


    ## Always show the side image to match desktop layout across platforms.
    add SideImage() xalign 0.0 yalign 1.0


## Make the namebox available for styling through the Character object.
init python:
    config.character_id_prefixes.append('namebox')

style window is default
style say_label is default
style say_dialogue is default
style say_thought is say_dialogue

style namebox is default
style namebox_label is say_label

# New style for the "NAME:" label
style name_label:
    color "#666666"
    font "gui/CreteRound-Regular.ttf"  # Updated path to the actual font location
    size 20
    xpos 23
    ypos 20  # Changed from 25 to 20 to move it up by 5 pixels
    bold False

style narrator_name:
    color "#8899AA"  # Subtle blue-gray color
    xpos 95
    ypos gui.name_ypos  # Changed to use the same ypos as character names
    size gui.name_text_size

style window:
    xalign 0.5
    yalign gui.textbox_yalign
    ypos 0.95
    ysize gui.textbox_height
    xsize 1200  # Fixed width matching the SVG size

    # Use the SVG as the background
    background "gui/boardingpass_textbox.svg"

style say_label:
    properties gui.text_properties("name")
    xpos 95  # Moved to appear after the "NAME:" label
    ypos gui.name_ypos
    xanchor gui.name_xalign
    yanchor 0.0

style say_dialogue:
    properties gui.text_properties("dialogue")

    xpos gui.dialogue_xpos
    xsize gui.dialogue_width
    ypos gui.dialogue_ypos
    xalign 0.0  # Changed from xanchor to xalign for consistency
    yalign 0.0

    adjust_spacing False

## Input screen ################################################################
##
## This screen is used to display renpy.input. The prompt parameter is used to
## pass a text prompt in.
##
## This screen must create an input displayable with id "input" to accept the
## various input parameters.
##
## https://www.renpy.org/doc/html/screen_special.html#input

screen input(prompt):
    style_prefix "input"

    window:

        vbox:
            xanchor gui.dialogue_text_xalign
            xpos gui.dialogue_xpos
            xsize gui.dialogue_width
            ypos gui.dialogue_ypos

            text prompt style "input_prompt"
            input id "input"

style input_prompt is default

style input_prompt:
    xalign gui.dialogue_text_xalign
    properties gui.text_properties("input_prompt")

style input:
    xalign gui.dialogue_text_xalign
    xmaximum gui.dialogue_width


## Choice screen ###############################################################
##
## This screen is used to display the in-game choices presented by the menu
## statement. The one parameter, items, is a list of objects, each with caption
## and action fields.
##
## https://www.renpy.org/doc/html/screen_special.html#choice

screen choice(items):
    style_prefix "choice"

    vbox:
        for i in items:
            textbutton i.caption action i.action


style choice_vbox is vbox
style choice_button is default:
    properties gui.button_properties("choice_button")
    background "#222222EE"  # Darker, more opaque background
    hover_background "#333333EE"  # Slightly lighter when hovered
    padding (10, 8)

style choice_button_text is default:
    properties gui.text_properties("choice_button")
    color "#ffffff"  # White text for non-selected
    hover_color "#aaffaa"  # Light green on hover

style choice_vbox:
    xalign 0.5
    ypos 405
    yanchor 0.5

    spacing gui.choice_spacing


## Quick Menu screen ###########################################################
##
## The quick menu is displayed in-game to provide easy access to the out-of-game
## menus.

screen quick_menu():

    ## Ensure this appears on top of other screens.
    zorder 100

    if quick_menu:

        hbox:
            style_prefix "quick"

            xalign 0.5
            yalign 1.0

            textbutton _("Back") action Rollback()
            textbutton _("History") action ShowMenu('history')
            textbutton _("Skip") action Skip() alternate Skip(fast=True, confirm=True)
            textbutton _("Auto") action Preference("auto-forward", "toggle")
            textbutton _("Save") action ShowMenu('save')
            textbutton _("Q.Save") action QuickSave()
            textbutton _("Q.Load") action QuickLoad()
            textbutton _("Prefs") action ShowMenu('preferences')


## This code ensures that the quick_menu screen is displayed in-game, whenever
## the player has not explicitly hidden the interface.
init python:
    config.overlay_screens.append("quick_menu")
    config.overlay_screens.append("miles_display")  # Add our miles display to overlay screens

default quick_menu = True

style quick_button is default
style quick_button_text is button_text

style quick_button:
    properties gui.button_properties("quick_button")

style quick_button_text:
    properties gui.text_properties("quick_button")


################################################################################
## Main and Game Menu Screens
################################################################################

## Navigation screen ###########################################################
##
## This screen is included in the main and game menus, and provides navigation
## to other menus, and to start the game.

screen navigation(active_screen=None):

    vbox:
        style_prefix "navigation"

        xpos gui.navigation_xpos
        yalign 0.5

        spacing gui.navigation_spacing

        if main_menu:

            textbutton _("Start") action Start()

        else:

            textbutton _("History"):
                action ShowMenu("history")
                # Highlight this button if we're on the history screen
                selected active_screen == _("History")

            textbutton _("Save"):
                action ShowMenu("save")
                # Highlight this button if we're on the save screen
                selected active_screen == _("Save")

        textbutton _("Load"):
            action ShowMenu("load")
            # Highlight this button if we're on the load screen
            selected active_screen == _("Load")

        textbutton _("Kickstarter✈️"):
            action OpenURL("https://www.kickstarter.com/projects/overwoodstudios/mile-high-maneuvers-by-overwood-studios?ref=8qjthf")

        textbutton _("Preferences"):
            action ShowMenu("preferences")
            # Highlight this button if we're on the preferences screen
            selected active_screen == _("Preferences")

        

        if _in_replay:

            textbutton _("End Replay") action EndReplay(confirm=True)

        elif not main_menu:

            textbutton _("Main Menu") action MainMenu()

        textbutton _("About"):
            action ShowMenu("about")
            # Highlight this button if we're on the about screen
            selected active_screen == _("About")

        if renpy.variant("pc") or (renpy.variant("web") and not renpy.variant("mobile")):

            ## Help isn't necessary or relevant to mobile devices.
            textbutton _("Help"):
                action ShowMenu("help")
                # Highlight this button if we're on the help screen
                selected active_screen == _("Help")

        if renpy.variant("pc"):

            ## The quit button is banned on iOS and unnecessary on Android and
            ## Web.
            textbutton _("Quit") action Quit(confirm=not main_menu)


style navigation_button is gui_button:
    background "#333333AA"  # Semi-transparent dark button
    hover_background "#444444AA"  # Slightly lighter when hovered
    selected_background "#333333AA"  # Same as normal background
    padding (15, 10)
    
style navigation_button_text is gui_button_text:
    color "#ffffff"
    hover_color "#aaffaa"  # Light green hover color
    selected_color "#aaffaa"  # Light green for selected buttons
    font "gui/CreteRound-Regular.ttf"


## Main Menu screen ############################################################
##
## Used to display the main menu when Ren'Py starts.
##
## https://www.renpy.org/doc/html/screen_special.html#main-menu

screen main_menu():

    ## This ensures that any other menu screen is replaced.
    tag menu
    
    ## Play main menu music if not already playing
    on "show" action If(not renpy.music.is_playing(channel='music'),
                        Play("music", "sounds/main_menu_ambiance_rev2.ogg", fadein=2.0, loop=True))

    add gui.main_menu_background

    ## This empty frame darkens the main menu.
    frame:
        style "main_menu_frame"

    ## The use statement includes another screen inside this one. The actual
    ## contents of the main menu are in the navigation screen.
    use navigation

    if gui.show_name:

        vbox:
            style "main_menu_vbox"

            text "[config.name!t]":
                style "main_menu_title"

            text "[config.version]":
                style "main_menu_version"


style main_menu_frame is empty
style main_menu_vbox is vbox
style main_menu_text is gui_text
style main_menu_title is main_menu_text
style main_menu_version is main_menu_text

style main_menu_frame:
    xsize 315  # Reduced by 25% from 420
    yfill True
    background "#222222AA"  # Semi-transparent dark grey background

style main_menu_vbox:
    xalign 1.0
    xoffset -30
    xmaximum 1200
    yalign 1.0
    yoffset -30

style main_menu_text:
    properties gui.text_properties("main_menu", accent=True)

style main_menu_title:
    properties gui.text_properties("title")
    color "#aaffaa"  # Light green title

style main_menu_version:
    properties gui.text_properties("version")
    color "#ffffff"  # White version text


## Game Menu screen ############################################################
##
## This lays out the basic common structure of a game menu screen. It's called
## with the screen title, and displays the background, title, and navigation.
##
## The scroll parameter can be None, or one of "viewport" or "vpgrid".
## This screen is intended to be used with one or more children, which are
## transcluded (placed) inside it.

screen game_menu(title, scroll=None, yinitial=0.0, spacing=0):

    style_prefix "game_menu"

    if main_menu:
        add "#000000AA"  # Semi-transparent black background for main menu
    else:
        add "#000000AA"  # Semi-transparent black background for game menu

    frame:
        style "game_menu_outer_frame"

        hbox:

            ## Reserve space for the navigation section.
            frame:
                style "game_menu_navigation_frame"

            frame:
                style "game_menu_content_frame"

                if scroll == "viewport":

                    viewport:
                        yinitial yinitial
                        scrollbars "vertical"
                        mousewheel True
                        draggable True
                        pagekeys True

                        side_yfill True

                        vbox:
                            spacing spacing

                            transclude

                elif scroll == "vpgrid":

                    vpgrid:
                        cols 1
                        yinitial yinitial

                        scrollbars "vertical"
                        mousewheel True
                        draggable True
                        pagekeys True

                        side_yfill True

                        spacing spacing

                        transclude

                else:

                    transclude

    # Use navigation with the current screen name passed as a parameter
    use navigation(title)

    textbutton _("Return"):
        style "return_button"
        action Return()

    if main_menu:
        key "game_menu" action ShowMenu("main_menu")


style game_menu_outer_frame is empty
style game_menu_navigation_frame is empty
style game_menu_content_frame is empty
style game_menu_viewport is gui_viewport
style game_menu_side is gui_side
style game_menu_scrollbar is gui_vscrollbar

style game_menu_label is gui_label
style game_menu_label_text is gui_label_text

style return_button is navigation_button
style return_button_text is navigation_button_text

style game_menu_outer_frame:
    bottom_padding 45
    top_padding 180
    background "#222222AA"  # Semi-transparent dark grey

style game_menu_navigation_frame:
    xsize 315  # Reduced by 25% from 420 to match main menu
    yfill True
    background "#1a1a1aAA"  # Slightly darker semi-transparent background

style game_menu_content_frame:
    left_margin 60
    right_margin 30
    top_margin 15
    background "#1a1a1aAA"  # Slightly darker semi-transparent background

style game_menu_viewport:
    xsize 1380

style game_menu_vscrollbar:
    unscrollable gui.unscrollable

style game_menu_side:
    spacing 15

style game_menu_label:
    xpos 75
    ysize 180

style game_menu_label_text:
    size gui.title_text_size
    color "#aaffaa"  # Light green color for titles
    yalign 0.5
    font "gui/CreteRound-Regular.ttf"

style return_button:
    xpos gui.navigation_xpos
    yalign 1.0
    yoffset -45
    
style return_button_text:
    color "#ffffff"
    hover_color "#aaffaa"  # Light green hover color
    
# Update navigation button styles
style navigation_button is gui_button:
    background "#333333AA"  # Semi-transparent dark button
    hover_background "#444444AA"  # Slightly lighter when hovered
    selected_background "#333333AA"  # Same as normal background
    padding (15, 10)
    
style navigation_button_text is gui_button_text:
    color "#ffffff"
    hover_color "#aaffaa"  # Light green hover color
    selected_color "#aaffaa"  # Light green for selected buttons
    font "gui/CreteRound-Regular.ttf"



style miles_frame is frame:
    xalign 0.5
    yalign 0.5
    xpadding 50
    ypadding 30
    background "#222222AA" # Semi-transparent dark grey background (same as top bar)

style miles_label is text:
    size 28
    color "#ffffff"
    xalign 0.0 # Align labels left

style miles_value is text:
    size 28
    color "#aaffaa" # Greenish color for values
    xalign 1.0 # Align values right

style miles_final_value is miles_value:
    color "#ffffaa" # Yellowish for the animated value
    min_width 150 # Give it some fixed width to avoid layout jumps

style miles_button is button:
    xalign 0.5

style miles_button_text is button_text:
    size 24
    hover_color "#ffddaa"


## About screen ################################################################
##
## This screen gives credit and copyright information about the game and Ren'Py.
##
## There's nothing special about this screen, and hence it also serves as an
## example of how to make a custom screen.

screen about():

    tag menu

    ## This use statement includes the game_menu screen inside this one. The
    ## vbox child is then included inside the viewport inside the game_menu
    ## screen.
    use game_menu(_("About"), scroll="viewport"):

        style_prefix "about"

        vbox:

            label "[config.name!t]"
            text _("Version [config.version!t]\n")

            ## gui.about is usually set in options.rpy.
            if gui.about:
                text "[gui.about!t]\n"

            text _("Made with {a=https://www.renpy.org/}Ren'Py{/a} [renpy.version_only].\n\n[renpy.license!t]")
            text _("")
            text _("Character Art by {a=https://x.com/ArteJaxton}ArteJaxton{/a}")
            text _("Background Illustrations by {a=https://x.com/Russelfur}Russelfur{/a}")
            text _("")
            text _("BARQ! name and likeness used with permission. Download {a=https://barq.app/}BARQ!{/a} - a social app for furries by furries.")


style about_label is gui_label
style about_label_text is gui_label_text
style about_text is gui_text

style about_label_text:
    size gui.label_text_size


## Load and Save screens #######################################################
##
## These screens are responsible for letting the player save the game and load
## it again. Since they share nearly everything in common, both are implemented
## in terms of a third screen, file_slots.
##
## https://www.renpy.org/doc/html/screen_special.html#save https://
## www.renpy.org/doc/html/screen_special.html#load

screen save():

    tag menu

    use file_slots(_("Save"))


screen load():

    tag menu

    use file_slots(_("Load"))


screen file_slots(title):

    default page_name_value = FilePageNameInputValue(pattern=_("Page {}"), auto=_("Automatic saves"), quick=_("Quick saves"))

    use game_menu(title):

        fixed:

            ## This ensures the input will get the enter event before any of the
            ## buttons do.
            order_reverse True

            ## The page name, which can be edited by clicking on a button.
            button:
                style "page_label"

                key_events True
                xalign 0.5
                action page_name_value.Toggle()

                input:
                    style "page_label_text"
                    value page_name_value

            ## The grid of file slots.
            grid gui.file_slot_cols gui.file_slot_rows:
                style_prefix "slot"

                xalign 0.5
                yalign 0.5

                spacing gui.slot_spacing

                for i in range(gui.file_slot_cols * gui.file_slot_rows):

                    $ slot = i + 1

                    button:
                        # Use custom action for loading to stop music, regular FileAction for saving
                        if CurrentScreenName() == "load":
                            action LoadWithMusicStop(slot)
                        else:
                            action FileAction(slot)

                        has vbox

                        add FileScreenshot(slot) xalign 0.5

                        text FileTime(slot, format=_("{#file_time}%A, %B %d %Y, %H:%M"), empty=_("empty slot")):
                            style "slot_time_text"

                        text FileSaveName(slot):
                            style "slot_name_text"

                        key "save_delete" action FileDelete(slot)

            ## Buttons to access other pages.
            vbox:
                style_prefix "page"

                xalign 0.5
                yalign 1.0

                hbox:
                    xalign 0.5

                    spacing gui.page_spacing

                    textbutton _("<") action FilePagePrevious()

                    if config.has_autosave:
                        textbutton _("{#auto_page}A") action FilePage("auto")

                    if config.has_quicksave:
                        textbutton _("{#quick_page}Q") action FilePage("quick")

                    ## range(1, 10) gives the numbers from 1 to 9.
                    for page in range(1, 10):
                        textbutton "[page]" action FilePage(page)

                if config.has_sync:
                    if CurrentScreenName() == "save":
                        textbutton _("Upload Sync"):
                            action UploadSync()
                            xalign 0.5
                    else:
                        textbutton _("Download Sync"):
                            action DownloadSync()
                            xalign 0.5


style page_label is gui_label
style page_label_text is gui_label_text
style page_button is gui_button
style page_button_text is gui_button_text

style slot_button is gui_button
style slot_button_text is gui_button_text
style slot_time_text is slot_button_text
style slot_name_text is slot_button_text

style page_label:
    xpadding 75
    ypadding 5

style page_label_text:
    textalign 0.5
    layout "subtitle"
    hover_color gui.hover_color

style page_button:
    properties gui.button_properties("page_button")

style page_button_text:
    properties gui.text_properties("page_button")

style slot_button:
    properties gui.button_properties("slot_button")

style slot_button_text:
    properties gui.text_properties("slot_button")
    
style slot_time_text:
    properties gui.text_properties("slot_button")
    
style slot_name_text:
    properties gui.text_properties("slot_button")


## Preferences screen ##########################################################
##
## The preferences screen allows the player to configure the game to better suit
## themselves.
##
## https://www.renpy.org/doc/html/screen_special.html#preferences

screen preferences():

    tag menu

    use game_menu(_("Preferences"), scroll="viewport"):

        vbox:

            hbox:
                box_wrap True

                if renpy.variant("pc") or renpy.variant("web"):

                    vbox:
                        style_prefix "radio"
                        label _("Display")
                        textbutton _("Window") action Preference("display", "window")
                        textbutton _("Fullscreen") action Preference("display", "fullscreen")

                vbox:
                    style_prefix "check"
                    label _("Skip")
                    textbutton _("Unseen Text") action Preference("skip", "toggle")
                    textbutton _("After Choices") action Preference("after choices", "toggle")
                    textbutton _("Transitions") action InvertSelected(Preference("transitions", "toggle"))

                ## Additional vboxes of type "radio_pref" or "check_pref" can be
                ## added here, to add additional creator-defined preferences.

            null height (4 * gui.pref_spacing)

            hbox:
                style_prefix "slider"
                box_wrap True

                vbox:

                    label _("Text Speed")

                    bar value Preference("text speed")

                    label _("Auto-Forward Time")

                    bar value Preference("auto-forward time")

                vbox:

                    if config.has_music:
                        label _("Music Volume")

                        hbox:
                            bar value Preference("music volume")

                    if config.has_sound:

                        label _("Sound Volume")

                        hbox:
                            bar value Preference("sound volume")

                            if config.sample_sound:
                                textbutton _("Test") action Play("sound", config.sample_sound)


                    if config.has_voice:
                        label _("Voice Volume")

                        hbox:
                            bar value Preference("voice volume")

                            if config.sample_voice:
                                textbutton _("Test") action Play("voice", config.sample_voice)

                    if config.has_music or config.has_sound or config.has_voice:
                        null height gui.pref_spacing

                        textbutton _("Mute All"):
                            action Preference("all mute", "toggle")
                            style "mute_all_button"


style pref_label is gui_label
style pref_label_text is gui_label_text
style pref_vbox is vbox

style radio_label is pref_label
style radio_label_text is pref_label_text
style radio_button is gui_button
style radio_button_text is gui_button_text
style radio_vbox is pref_vbox

style check_label is pref_label
style check_label_text is pref_label_text
style check_button is gui_button
style check_button_text is gui_button_text
style check_vbox is pref_vbox

style slider_label is pref_label
style slider_label_text is pref_label_text
style slider_slider is gui_slider
style slider_button is gui_button
style slider_button_text is gui_button_text
style slider_pref_vbox is pref_vbox

style mute_all_button is check_button
style mute_all_button_text is check_button_text

style pref_label:
    top_margin gui.pref_spacing
    bottom_margin 3
    color "#aaffaa"  # Light green for labels
    font "gui/CreteRound-Regular.ttf"

style pref_label_text:
    yalign 1.0

style pref_vbox:
    xsize 338

style radio_vbox:
    spacing gui.pref_button_spacing

style radio_button:
    properties gui.button_properties("radio_button")
    foreground "gui/button/radio_[prefix_]foreground.png"
    background "#333333AA"  # Semi-transparent dark background
    hover_background "#444444AA"  # Slightly lighter when hovered
    
style radio_button_text:
    properties gui.text_properties("radio_button")
    color "#ffffff"  # White text
    hover_color "#aaffaa"  # Light green on hover
    font "gui/CreteRound-Regular.ttf"

style check_vbox:
    spacing gui.pref_button_spacing

style check_button:
    properties gui.button_properties("check_button")
    foreground "gui/button/check_[prefix_]foreground.png"
    background "#333333AA"  # Semi-transparent dark background
    hover_background "#444444AA"  # Slightly lighter when hovered
    
style check_button_text:
    properties gui.text_properties("check_button")
    color "#ffffff"  # White text
    hover_color "#aaffaa"  # Light green on hover
    font "gui/CreteRound-Regular.ttf"

style slider_slider:
    xsize 525

style slider_button:
    properties gui.button_properties("slider_button")
    yalign 0.5
    left_margin 15

style slider_button_text:
    properties gui.text_properties("slider_button")

style slider_vbox:
    xsize 675


## History screen ##############################################################
##
## This is a screen that displays the dialogue history to the player. While
## there isn't anything special about this screen, it does have to access the
## dialogue history stored in _history_list.
##
## https://www.renpy.org/doc/html/history.html

screen history():

    tag menu

    ## Avoid predicting this screen, as it can be very large.
    predict False

    use game_menu(_("History"), scroll=("vpgrid" if gui.history_height else "viewport"), yinitial=1.0, spacing=gui.history_spacing):

        style_prefix "history"

        for h in _history_list:

            window:

                ## This lays things out properly if history_height is None.
                has fixed:
                    yfit True

                if h.who:

                    label h.who:
                        style "history_name"
                        substitute False

                        ## Take the color of the who text from the Character, if
                        ## set.
                        if "color" in h.who_args:
                            text_color h.who_args["color"]

                $ what = renpy.filter_text_tags(h.what, allow=gui.history_allow_tags)
                text what:
                    substitute False

        if not _history_list:
            label _("The dialogue history is empty.")


## This determines what tags are allowed to be displayed on the history screen.

define gui.history_allow_tags = { "alt", "noalt", "rt", "rb", "art" }


style history_window is empty

style history_name is gui_label
style history_name_text is gui_label_text
style history_text is gui_text

style history_label is gui_label
style history_label_text is gui_label_text

style history_window:
    xfill True
    ysize gui.history_height

style history_name:
    xpos gui.history_name_xpos
    xanchor gui.history_name_xalign
    ypos gui.history_name_ypos
    xsize gui.history_name_width

style history_name_text:
    min_width gui.history_name_width
    textalign gui.history_name_xalign

style history_text:
    xpos gui.history_text_xpos
    ypos gui.history_text_ypos
    xanchor gui.history_text_xalign
    xsize gui.history_text_width
    min_width gui.history_text_width
    textalign gui.history_text_xalign
    layout ("subtitle" if gui.history_text_xalign else "tex")

style history_label:
    xfill True

style history_label_text:
    xalign 0.5


## Help screen #################################################################
##
## A screen that gives information about key and mouse bindings. It uses other
## screens (keyboard_help, mouse_help, and gamepad_help) to display the actual
## help.

screen help():

    tag menu

    default device = "keyboard"

    use game_menu(_("Help"), scroll="viewport"):

        style_prefix "help"

        vbox:
            spacing 23

            hbox:

                textbutton _("Keyboard") action SetScreenVariable("device", "keyboard")
                textbutton _("Mouse") action SetScreenVariable("device", "mouse")

                if GamepadExists():
                    textbutton _("Gamepad") action SetScreenVariable("device", "gamepad")

            if device == "keyboard":
                use keyboard_help
            elif device == "mouse":
                use mouse_help
            elif device == "gamepad":
                use gamepad_help


screen keyboard_help():

    hbox:
        label _("Enter")
        text _("Advances dialogue and activates the interface.")

    hbox:
        label _("Space")
        text _("Advances dialogue without selecting choices.")

    hbox:
        label _("Arrow Keys")
        text _("Navigate the interface.")

    hbox:
        label _("Escape")
        text _("Accesses the game menu.")

    hbox:
        label _("Ctrl")
        text _("Skips dialogue while held down.")

    hbox:
        label _("Tab")
        text _("Toggles dialogue skipping.")

    hbox:
        label _("Page Up")
        text _("Rolls back to earlier dialogue.")

    hbox:
        label _("Page Down")
        text _("Rolls forward to later dialogue.")

    hbox:
        label "H"
        text _("Hides the user interface.")

    hbox:
        label "S"
        text _("Takes a screenshot.")

    hbox:
        label "V"
        text _("Toggles assistive {a=https://www.renpy.org/l/voicing}self-voicing{/a}.")

    hbox:
        label "Shift+A"
        text _("Opens the accessibility menu.")


screen mouse_help():

    hbox:
        label _("Left Click")
        text _("Advances dialogue and activates the interface.")

    hbox:
        label _("Middle Click")
        text _("Hides the user interface.")

    hbox:
        label _("Right Click")
        text _("Accesses the game menu.")

    hbox:
        label _("Mouse Wheel Up")
        text _("Rolls back to earlier dialogue.")

    hbox:
        label _("Mouse Wheel Down")
        text _("Rolls forward to later dialogue.")


screen gamepad_help():

    hbox:
        label _("Right Trigger\nA/Bottom Button")
        text _("Advances dialogue and activates the interface.")

    hbox:
        label _("Left Trigger\nLeft Shoulder")
        text _("Rolls back to earlier dialogue.")

    hbox:
        label _("Right Shoulder")
        text _("Rolls forward to later dialogue.")

    hbox:
        label _("D-Pad, Sticks")
        text _("Navigate the interface.")

    hbox:
        label _("Start, Guide, B/Right Button")
        text _("Accesses the game menu.")

    hbox:
        label _("Y/Top Button")
        text _("Hides the user interface.")

    textbutton _("Calibrate") action GamepadCalibrate()


style help_button is gui_button
style help_button_text is gui_button_text
style help_label is gui_label
style help_label_text is gui_label_text
style help_text is gui_text

style help_button:
    properties gui.button_properties("help_button")
    xmargin 12

style help_button_text:
    properties gui.text_properties("help_button")

style help_label:
    xsize 375
    right_padding 30

style help_label_text:
    size gui.text_size
    xalign 1.0
    textalign 1.0



################################################################################
## Additional screens
################################################################################


## Confirm screen ##############################################################
##
## The confirm screen is called when Ren'Py wants to ask the player a yes or no
## question.
##
## https://www.renpy.org/doc/html/screen_special.html#confirm

# Add a default for tracking if minigames are active and if first day has started
default first_day_started = False
default persistent.in_minigame = False

screen confirm(message, yes_action, no_action):

    ## Ensure other screens do not get input while this screen is displayed.
    modal True

    zorder 200

    style_prefix "confirm"

    add "gui/overlay/confirm.png"

    frame:

        vbox:
            xalign .5
            yalign .5
            spacing 45

            label _(message):
                style "confirm_prompt"
                xalign 0.5

            hbox:
                xalign 0.5
                spacing 150

                textbutton _("Yes") action yes_action
                textbutton _("No") action no_action

    ## Right-click and escape answer "no".
    key "game_menu" action no_action


style confirm_frame is gui_frame
style confirm_prompt is gui_prompt
style confirm_prompt_text is gui_prompt_text
style confirm_button is gui_medium_button
style confirm_button_text is gui_medium_button_text

style confirm_frame:
    background Frame([ "gui/confirm_frame.png", "gui/frame.png"], gui.confirm_frame_borders, tile=gui.frame_tile)
    padding gui.confirm_frame_borders.padding
    xalign .5
    yalign .5

style confirm_prompt_text:
    textalign 0.5
    layout "subtitle"
    color "#333333"  # Dark text for prompt

style confirm_button:
    properties gui.button_properties("confirm_button")
    background "#222222EE"  # Darker background to match our theme
    hover_background "#333333EE"  # Slightly lighter when hovered

style confirm_button_text:
    properties gui.text_properties("confirm_button")
    color "#ffffff"  # White text
    hover_color "#aaffaa"  # Light green on hover
    font "gui/CreteRound-Regular.ttf"


## Miles display overlay in top left corner
screen miles_display():
    # Only show this if the first day has started and we're not in a minigame
    if first_day_started and not persistent.in_minigame:
        zorder 100  # Keep it on top
        
        # Dark grey bar across the top 5% of the screen
        frame:
            style "top_bar_frame"
        
        # Status information positioned on the bar
        frame:
            style "status_display_frame"
            hbox:
                spacing 30  # Add more space between items
                
                # Day
                hbox:
                    spacing 10
                    text "Day:" style "status_display_label"
                    text "[mc.day]" style "status_display_value"
                
                # Level
                hbox:
                    spacing 10
                    text "Level:" style "status_display_label"
                    text "[mc.level]" style "status_display_value"
                
                # Miles
                hbox:
                    spacing 10
                    text "Miles:" style "status_display_label"
                    text "[mc.miles]" style "status_display_value"
                
                # Aircraft Assignment
                hbox:
                    spacing 10
                    text "Aircraft:" style "status_display_label"
                    text "[mc.aircraft_type]" style "status_display_value"
        
        # Stats button on the right side of the bar
        frame:
            style "stats_button_frame"
            xalign 1.0
            ypos 0
            textbutton "Stats":
                # Use the custom RefreshPlayerStats action for proper refresh
                action RefreshPlayerStats() 
                style "top_bar_button"
                text_style "top_bar_button_text"

# Style definitions for the stats button and stats screen
style top_bar_frame:
    background "#222222AA"  # Semi-transparent dark grey
    xsize 1920  # Full screen width
    ysize 54    # 5% of screen height (1080 * 0.05 = 54)
    xpos 0
    ypos 0

style status_display_frame:
    background None  # No background as it sits on the bar
    padding (20, 10)
    xpos 20
    ypos 0  # Align to top
    
style status_display_label:
    color "#ffffff"
    size 28
    font "gui/CreteRound-Regular.ttf"
    
style status_display_value:
    color "#aaffaa"  # Light green for values
    size 28
    font "gui/CreteRound-Regular.ttf"
    min_width 40  # Ensure enough room for values

style top_bar_button:
    background "#444444"
    hover_background "#666666"
    padding (15, 5)
    xminimum 100

style top_bar_button_text:
    color "#ffffff"
    hover_color "#aaffaa"  # Same light green as the miles value when hovered
    size 28
    font "gui/CreteRound-Regular.ttf"

style stats_button_frame:
    background None
    padding (5, 2)
    ysize 54  # Match the top bar height

style stats_overlay_background:
    background "#00000099"  # Semi-transparent black
    xfill True
    yfill True

style stats_panel:
    background "#222222EE"
    padding (30, 30)
    xsize 800
    ysize 680
    xalign 0.5
    yalign 0.5
    
style stats_header:
    xfill True
    padding (0, 0, 0, 10)
    background "#333333"
    
style stats_section:
    background "#333333CC"
    padding (20, 15)
    margin (0, 10)
    xfill True
    
style stats_section_header:
    color "#AAAAFF"
    size 24
    xalign 0.5

style stats_primary:
    color "#FFFFFF"
    size 22
    
style stats_description:
    color "#AAAAAA"
    size 16
    italic True
    
style stats_skill_label:
    color "#FFFFFF"
    size 22
    min_width 150
    
style stats_skill_value:
    color "#AAFFAA"
    size 22
    bold True
    
style stats_progress:
    color "#FFFFAA"
    size 22
    
style stats_close_button:
    background "#444444"
    hover_background "#666666"
    padding (20, 10)
    xsize 200
    ypos 10
    xalign 0.5

style stats_close_button_text:
    color "#ffffff"
    hover_color "#aaffaa"
    size 24
    xalign 0.5






#CUSTOM SCREENS

screen character_creation:
    style_prefix "character_creation"
    
    frame:
        xsize 1200
        ysize 600
        xalign 0.5
        yalign 0.5
        padding (0, 0)
        background "gui/boardingpass_character.svg"
        
        # Character creation header
        text "CHARACTER CREATION" style "cc_header"
        
        # Name field
        text "NAME:" style "cc_name_label"
        text "[mc.name]" style "cc_name_value"
        
        # Points Remaining label and value - repositioned
        text "POINTS REMAINING:" style "cc_points_label"
        text "[mc.points]" style "cc_points_value"
        
        # Stats area - stats positioned to center between the dotted lines
        for i, (stat_name, stat_var) in enumerate([
            ("Agility", "agility"),
            ("Charisma", "charisma"),
            ("Intellect", "intellect")
        ]):
            # Compute vertical center between the dashed lines for this row
            $ row_center = 180 + (i * 70) + 35
            
            # Stat name
            text stat_name:
                style "cc_stat_name"
                xpos 50
                ypos row_center
                yanchor 0.5
            
            # Stat value - positioned 200px to the left of its original position
            text str(getattr(mc, stat_var)):
                style "cc_stat_value"
                xpos 300  # Reduced from ~500 (was 50+180+150+null width) to 300
                ypos row_center
                yanchor 0.5
            
            # Stat description - now with fixed width to force 2 lines
            text stat_descriptions.get(stat_var, ""):
                style "cc_stat_desc"
                xpos 650
                ypos row_center
                yanchor 0.5
                xsize 400  # Fixed width to force text wrapping
            
            # Plus/Minus buttons
            hbox:
                xpos 1050
                ypos row_center
                yanchor 0.5
                spacing 10
                
                textbutton "-":
                    style "cc_button"
                    sensitive getattr(mc, stat_var) > 0
                    action [
                        SetField(mc, stat_var, getattr(mc, stat_var) - 1),
                        SetField(mc, "points", mc.points + 1)
                    ]
                
                textbutton "+":
                    style "cc_button"
                    sensitive mc.points > 0
                    action [
                        SetField(mc, stat_var, getattr(mc, stat_var) + 1),
                        SetField(mc, "points", mc.points - 1)
                    ]
        
        # Bottom buttons - positioned parallel with the barcode
        hbox:
            xpos 600  # Center of the card, which is 1200px wide
            ypos 550  # Same y position as the barcode
            spacing 50
            
            textbutton "Reset":
                style "cc_action_button"
                action Function(reset_stats, mc)
            
            textbutton "Confirm":  # Changed from "Continue" to "Confirm"
                style "cc_action_button"
                sensitive mc.points == 0
                action Return()

# Character creation styles
style character_creation_frame:
    background None

style cc_header:
    font "gui/CreteRound-Regular.ttf"
    size 30
    color "#666666"
    xpos 25
    ypos 45

style cc_name_label:
    font "gui/CreteRound-Regular.ttf"
    size 20
    color "#666666"
    xpos 25
    ypos 110

style cc_name_value:
    font "gui/CreteRound-Regular.ttf"
    size 24
    color "#000066"
    bold True
    xpos 95
    ypos 110

style cc_points_label:
    font "gui/CreteRound-Regular.ttf"
    size 20
    color "#666666"
    xpos 800  # Shifted left to be next to the box
    ypos 105  # Shifted up 5 pixels from 110 to 105

style cc_points_value:
    font "gui/CreteRound-Regular.ttf"
    size 22
    color "#000066"
    bold True
    xpos 1080  # Center of the box
    ypos 110   # Shifted down 2 more pixels from 108 to 110
    xanchor 0.5  # Anchor from the center horizontally
    yanchor 0.5  # Anchor from the center vertically
    min_width 60  # Ensure consistent width regardless of content

style cc_stat_name:
    font "gui/CreteRound-Regular.ttf"
    size 28
    color "#000066"
    bold True
    min_width 180
    xalign 0.0

style cc_stat_value:
    font "gui/CreteRound-Regular.ttf"
    size 34
    color "#000066"
    bold True
    xalign 0.0

style cc_stat_desc:
    font "gui/CreteRound-Regular.ttf"
    size 18
    color "#666666"
    italic True
    line_spacing 2
    xalign 0.0
    layout "subtitle"
    text_align 0.0

style cc_button:
    font "gui/CreteRound-Regular.ttf"
    size 24
    color "#FFFFFF"
    xsize 40
    ysize 40
    background "#cccccc"
    hover_background "#6684a3"
    insensitive_background "#EEEEEE"
    insensitive_color "#FFFFFF"

style cc_action_button:
    font "gui/CreteRound-Regular.ttf"
    size 20
    color "#FFFFFF"
    xsize 150
    ysize 30
    background "#6684a3"
    hover_background "#003366"
    insensitive_background "#CCCCCC"
    insensitive_color "#FFFFFF"



screen miles_summary_screen(starting_balance, raw_miles_earned, skill_name=None):
    # Displays the miles earned and animates the final total.
    #
    # Args:
    #     starting_balance (int): Miles the player had before earning more.
    #     raw_miles_earned (int): Miles earned before skill multiplier.
    #     skill_name (str, optional): Name of the skill providing the multiplier. None for no skill bonus.
    $ renpy.hide_screen("say")
    modal True # Prevent clicking outside the screen

    style_prefix "miles"

    # Look up skill value and calculate multiplier if a skill is provided
    if skill_name:
        $ skill_value = getattr(mc, skill_name, 0)  # Get attribute, default 0 if not found
        $ skill_display_name = skill_name.capitalize()  # For display
        $ skill_multiplier = 1 + (skill_value / 10.0)
        $ miles_with_multiplier = int(raw_miles_earned * skill_multiplier)
    else:
        # No skill multiplier
        $ miles_with_multiplier = raw_miles_earned
    
    $ final_balance = starting_balance + miles_with_multiplier
    
    # Screen variables for animation control
    default displayed_balance = starting_balance
    default animation_duration = 1.5  # Total seconds the animation should take
    default update_interval = 0.04    # How often to update (seconds)
    default steps = animation_duration / update_interval  # Total update steps
    default increment_per_step = max(1, miles_with_multiplier / steps)  # Miles per step

    # The main frame
    frame:
        # Use a vbox for vertical layout
        vbox:
            spacing 15 # Space between elements

            # Use an hbox for label + value pairs
            hbox:
                text "Starting Balance :" style "miles_label"
                text "[starting_balance]" style "miles_value"

            hbox:
                text "Base Miles :" style "miles_label"
                text "+[raw_miles_earned]" style "miles_value"
                
            # Display skill multiplier only if a skill is provided
            if skill_name:
                hbox:
                    text "[skill_display_name] Multiplier: " style "miles_label"
                    text "x[skill_multiplier:.1f]" style "miles_value"
                
                # Display miles with multiplier
                hbox:
                    text "Total Earned :" style "miles_label"
                    text "+[miles_with_multiplier]" style "miles_value"

            # Separator
            null height 10 # Add a bit of space

            # The animated final balance
            hbox:
                text "New Balance: " style "miles_label"
                text "[int(displayed_balance)]" style "miles_final_value" # Display as integer to remove decimals

            # Separator
            null height 20

            # Button to close the screen
            button:
                style "miles_continue_button"
                text "Continue" style "miles_continue_button_text"
                action Return() # Returns control to the script after 'call screen'

    # Timer to update the displayed_balance
    # Uses fixed duration animation regardless of miles amount
    timer update_interval repeat True action If(
        displayed_balance < final_balance,
        SetScreenVariable("displayed_balance", 
            min(final_balance, displayed_balance + increment_per_step)),
        NullAction()
    )

# Add styles for miles continue button
style miles_continue_button:
    xalign 0.5
    background "#444444"
    hover_background "#666666"
    padding (20, 10)
    xsize 200

style miles_continue_button_text:
    color "#aaffaa"  # Same green as the status values
    size 24
    xalign 0.5


screen persistent_stats_button():
    zorder 100  # Make sure it appears above other screens
    
    frame:
        style "stats_button_frame"
        xalign 1.0
        yalign 0.0
        padding (10, 5)
        
        textbutton "Stats":
            action Show("player_stats")
            style "stats_button"


screen player_stats(refresh_time=0):
    modal True  # Prevent interaction with other elements
    zorder 200  # Ensure it appears above the say screen
    style_prefix "player_stats"
    
    frame:
        xfill True
        yfill True
        background "#000000aa"  # Semi-transparent background
        
        frame:
            xsize 1700  # Almost full 1920px width
            ymaximum 900  # Maximum height to prevent covering top bar
            xalign 0.5
            yalign 0.55  # Slightly below center to avoid top bar
            background "#1a1a1a"
            padding (40, 30)
            
            vbox:
                spacing 20
                xfill True
                
                # Title
                text "Player Stats" size 42 xalign 0.5 color "#aaffaa" bold True
                
                # Section 1: Player Basic Info (horizontal bar)
                frame:
                    xsize 1620
                    xalign 0.5
                    padding (30, 15)
                    background "#333333AA"
                    
                    hbox:
                        spacing 150
                        xalign 0.5
                        
                        hbox:
                            spacing 15
                            text "Name:" size 24 color "#ffffff"
                            text "[mc.name]" size 24 color "#aaffaa" bold True
                        
                        hbox:
                            spacing 15
                            text "Level:" size 24 color "#ffffff"
                            text "[mc.level]" size 24 color "#aaffaa" bold True
                        
                        hbox:
                            spacing 15
                            text "Miles:" size 24 color "#ffffff"
                            text "[mc.miles]" size 24 color "#aaffaa" bold True
                
                # Section 2: Core Attributes and Personality Scores (side by side)
                hbox:
                    spacing 20
                    xfill True
                    ysize 340  # Fixed height for container
                    
                    # Core Attributes (left side)
                    frame:
                        xsize 800
                        ysize 340  # Match container height
                        padding (30, 25)
                        background "#333333AA"
                        
                        # Use fixed positioning for perfect alignment
                        fixed:
                            # Header
                            text "Core Attributes" size 28 xalign 0.5 ypos 0 color "#aaffaa" bold True
                            
                            # Stats - using fixed positioning for perfect alignment
                            for i, (stat_name, stat_var) in enumerate([
                                ("Agility", "agility"),
                                ("Charisma", "charisma"),
                                ("Endurance", "endurance"),
                                ("Intellect", "intellect"),
                                ("Sexperience", "sexperience")
                            ]):
                                $ y_offset = 50 + (i * 48)  # Reduced row height to fit better
                                
                                # Stat name
                                text stat_name:
                                    size 22
                                    xpos 0
                                    ypos y_offset
                                    color "#ffffff"
                                
                                # Stat value - absolute positioning
                                text "[getattr(mc, stat_var)]":
                                    size 22
                                    xpos 150  # Fixed x position for all values
                                    ypos y_offset
                                    xanchor 0.5  # Center on this x position
                                    color "#aaffaa"
                                    bold True
                                
                                # Description
                                $ desc = stat_descriptions.get(stat_var, "")
                                text desc:
                                    size 16
                                    xpos 190
                                    ypos y_offset + 3  # Slight adjustment for visual alignment
                                    xsize 580
                                    color "#999999"
                    
                    # Personality Scores (right side)
                    frame:
                        xsize 800
                        ysize 340  # Match container height
                        padding (30, 25)
                        background "#333333AA"
                        
                        # Use fixed positioning for perfect alignment
                        fixed:
                            # Header
                            text "Personality Scores" size 28 xalign 0.5 ypos 0 color "#aaffaa" bold True
                            
                            # Personality traits - using fixed positioning
                            for i, (trait_name, trait_var, trait_desc) in enumerate([
                                ("Assertiveness", "assertiveness", "How dominant and confident you are"),
                                ("Submissiveness", "submissiveness", "How willing you are to submit"),
                                ("Humor", "humor", "Your ability to make others laugh"),
                                ("Flirtiness", "flirtiness", "How flirtatious and charming you are")
                            ]):
                                $ y_offset = 50 + (i * 48)  # Now aligned with Core Attributes
                                
                                # Trait name - with enough width to not cut off
                                text trait_name:
                                    size 22
                                    xpos 0
                                    ypos y_offset
                                    xsize 145  # Fixed width to prevent cutoff
                                    color "#ffffff"
                                
                                # Trait value - shifted right by 60 pixels total
                                text "[getattr(mc, trait_var)]":
                                    size 22
                                    xpos 210  # Shifted right by 60 pixels total (was 150)
                                    ypos y_offset
                                    xanchor 0.5  # Center on this x position
                                    color "#aaffaa"
                                    bold True
                                
                                # Description - shifted right by 60 pixels total
                                text trait_desc:
                                    size 16
                                    xpos 250  # Shifted right by 60 pixels total (was 190)
                                    ypos y_offset + 3
                                    xsize 520  # Further reduced width to fit after shift
                                    color "#999999"
                
                # Section 3: Sexual Skills (horizontal)
                frame:
                    xsize 1620
                    xalign 0.5
                    padding (30, 20)
                    background "#333333AA"
                    ysize 120  # Fixed height to prevent any overflow
                    
                    fixed:
                        # Header
                        text "Sexual Skills" size 28 xpos 820 ypos 0 xanchor 0.5 color "#aaffaa" bold True
                        
                        # Oral Skill - left side
                        text "Oral Skill":
                            size 22
                            xpos 200
                            ypos 50
                            color "#ffffff"
                        
                        text "[mc.get_rounded_skill('oral_skill')]":
                            size 22
                            xpos 360  # Aligned with other values
                            ypos 50
                            xanchor 0.5
                            color "#aaffaa"
                            bold True
                        
                        text mc.get_oral_description():
                            size 16
                            xpos 400
                            ypos 53
                            xsize 300
                            color "#999999"
                        
                        # Anal Skill - right side
                        text "Anal Skill":
                            size 22
                            xpos 900
                            ypos 50
                            color "#ffffff"
                        
                        text "[mc.get_rounded_skill('anal_skill')]":
                            size 22
                            xpos 1060  # Same relative position as oral skill
                            ypos 50
                            xanchor 0.5
                            color "#aaffaa"
                            bold True
                        
                        text mc.get_anal_description():
                            size 16
                            xpos 1100
                            ypos 53
                            xsize 300
                            color "#999999"
                
                # Close button at the bottom
                null height 5
                
                textbutton "Close":
                    action Hide("player_stats")
                    xalign 0.5
                    style "stats_close_button"
                    text_style "stats_close_button_text"

# Style for the MHM text in the corner
style mhm_label:
    color "#ffffff"  # White text
    font "gui/CreteRound-Regular.ttf"  # Changed from Arial to Crete Round
    size 13
    bold True
    xpos 1144
    ypos 25

################################################################################
## Day Transition Screen
################################################################################

screen day_transition():
    # Full screen modal that shows at the beginning of each day
    modal True
    zorder 200  # Ensure it's above other elements
    
    # Semi-transparent background covering entire screen
    frame:
        background "#000000AA"  # Semi-transparent black background
        xfill True
        yfill True
        
        # Main content container - styled like miles frame and notification screens
        frame:
            style_prefix "day_transition"
            background "#222222EE"  # Semi-transparent dark grey like the top bar
            xsize 800
            ysize 400
            xalign 0.5
            yalign 0.5
            padding (40, 40)
            
            vbox:
                spacing 30
                xalign 0.5
                yalign 0.5
                
                # Title text - now first
                text "MILE HIGH MANEUVERS":
                    size 36
                    xalign 0.5
                    color "#aaffaa"  # Light green like status values
                    font "gui/CreteRound-Regular.ttf"
                
                # Separator
                null height 20
                
                # Day header - now second
                text "DAY [mc.day]":
                    size 80
                    xalign 0.5
                    color "#ffffff"
                    font "gui/CreteRound-Regular.ttf"
                
                # Continue button at the bottom
                null height 40
                
                textbutton "Continue":
                    action Return()
                    xalign 0.5
                    style "stats_close_button"
                    text_style "stats_close_button_text"

# Styles for day transition screen - uses existing styles where possible
style day_transition_frame:
    background "#222222EE"  # Semi-transparent dark grey

style day_transition_text:
    font "gui/CreteRound-Regular.ttf"
    color "#ffffff"
    text_align 0.5
    xalign 0.5

################################################################################
## Disclaimer Screen
################################################################################

screen disclaimer_screen():
    modal True
    zorder 200
    
    # Black background covering entire screen
    frame:
        background "#000000"
        xfill True
        yfill True
        
        # Mid-grey semi-transparent box for content
        frame:
            style_prefix "disclaimer"
            background "#222222EE"
            xsize 900
            ysize 600
            xalign 0.5
            yalign 0.5
            padding (60, 50)
            
            vbox:
                spacing 30
                xfill True
                
                # Title
                text "MILE HIGH MANEUVERS DISCLAIMERS":
                    size 36
                    xalign 0.5
                    color "#aaffaa"  # Green color for heading
                    font "gui/CreteRound-Regular.ttf"
                    bold True
                    kerning 3.0  # Increased from 2.0 to 3.0
                
                # Age restriction
                vbox:
                    spacing 5
                    xfill True
                    text "AGE RESTRICTION:":
                        color "#aaffaa"  # Green color for section heading
                        font "gui/CreteRound-Regular.ttf"
                        bold True
                        size 24
                        kerning 3.0  # Increased from 2.0 to 3.0
                    text "You must be 18+ to play this game.":
                        color "#ffffff"  # White text
                        font "gui/CreteRound-Regular.ttf"
                        size 24
                
                # Content warning
                vbox:
                    spacing 5
                    xfill True
                    text "CONTENT WARNING:":
                        color "#aaffaa"  # Green color for section heading
                        font "gui/CreteRound-Regular.ttf"
                        bold True
                        size 24
                        kerning 3.0  # Increased from 2.0 to 3.0
                    text "This game contains explicit male/male sexual content and various kinks/fetishes that some players may find uncomfortable.":
                        color "#ffffff"  # White text
                        font "gui/CreteRound-Regular.ttf"
                        size 24
                
                # Likeness disclaimer
                vbox:
                    spacing 5
                    xfill True
                    text "LIKENESS DISCLAIMER:":
                        color "#aaffaa"  # Green color for section heading
                        font "gui/CreteRound-Regular.ttf"
                        bold True
                        size 24
                        kerning 3.0  # Increased from 2.0 to 3.0
                    text "All characters depicted are fictional and 18+ years of age. Any resemblance to real persons is coincidental.":
                        color "#ffffff"  # White text
                        font "gui/CreteRound-Regular.ttf"
                        size 24
                
                # Final statement
                text "This game is a work of fiction intended for adult audiences only.":
                    color "#ffffff"  # White text
                    font "gui/CreteRound-Regular.ttf"
                    size 24
                    xalign 0.5
                    line_spacing 5
                
                null height 20
                
                # Continue button
                textbutton "CONTINUE?":
                    action Return()
                    xalign 0.5
                    style "disclaimer_button"

style disclaimer_text:
    font "gui/CreteRound-Regular.ttf"
    color "#ffffff"
    size 24
    line_spacing 5

style disclaimer_button:
    xsize 250
    background "#444444"
    hover_background "#666666"
    padding (20, 12)

style disclaimer_button_text:
    font "gui/CreteRound-Regular.ttf"
    color "#ffffff"
    hover_color "#aaffaa"  # Light green on hover
    size 28
    xalign 0.5
    kerning 0.25  # Add letter spacing to button text

# Level Up Screen - similar to character creation but for leveling up
screen level_up_screen():
    style_prefix "character_creation"
    
    # Hide the say textbox when this screen appears
    $ renpy.hide_screen("say")
    on "show" action [ Hide("say"), SetVariable("_window_auto", False) ]
    on "hide" action SetVariable("_window_auto", True)
    
    # Make this screen modal and tag it
    modal True
    tag level_up
    
    default temp_stats = {"agility": 0, "charisma": 0, "intellect": 0}
    default points_to_allocate = 2
    default original_values = {"agility": mc.agility-1, "charisma": mc.charisma-1, "intellect": mc.intellect-1}
    
    frame:
        xsize 1200
        ysize 600
        xalign 0.5
        yalign 0.5
        padding (0, 0)
        background "gui/boardingpass_character.svg"
        
        # Level Up header
        text "LEVEL UP" style "cc_header"
        
        # Level display
        text "LEVEL:" style "cc_name_label"
        text "[mc.level]" style "cc_name_value"
        
        # Points Remaining label and value
        text "POINTS REMAINING:" style "cc_points_label"
        text "[points_to_allocate]" style "cc_points_value"
        
        # Stats area - stats positioned to center between the dotted lines
        for i, (stat_name, stat_var) in enumerate([
            ("Agility", "agility"),
            ("Charisma", "charisma"),
            ("Intellect", "intellect")
        ]):
            # Compute vertical center between the dashed lines for this row
            $ row_center = 180 + (i * 70) + 35
            
            # Stat name
            text stat_name:
                style "cc_stat_name"
                xpos 50
                ypos row_center
                yanchor 0.5
            
            # Stat value with formula: total = +1 + allocated points
            hbox:
                xpos 300
                ypos row_center
                yanchor 0.5
                spacing 5
                
                # Total value in dark blue - dynamically calculated
                $ dynamic_total = original_values[stat_var] + 1 + temp_stats[stat_var]
                text str(dynamic_total):
                    style "cc_stat_value"
                    xalign 0.0
                
                # Equal sign
                text " = " style "level_up_formula"
                
                # Auto +1 in green
                text "+1" style "level_up_auto_increment"
                
                # Plus sign
                text " + " style "level_up_formula"
                
                # Allocated points in light blue
                text str(temp_stats[stat_var]):
                    style "level_up_allocated"
                    xalign 0.0
            
            # Stat description
            text stat_descriptions.get(stat_var, ""):
                style "cc_stat_desc"
                xpos 650
                ypos row_center
                yanchor 0.5
                xsize 400
            
            # Plus/Minus buttons
            hbox:
                xpos 1050
                ypos row_center
                yanchor 0.5
                spacing 10
                
                textbutton "-":
                    style "cc_button"
                    sensitive temp_stats[stat_var] > 0
                    action [
                        SetDict(temp_stats, stat_var, temp_stats[stat_var] - 1),
                        SetScreenVariable("points_to_allocate", points_to_allocate + 1)
                    ]
                
                textbutton "+":
                    style "cc_button"
                    sensitive points_to_allocate > 0
                    action [
                        SetDict(temp_stats, stat_var, temp_stats[stat_var] + 1),
                        SetScreenVariable("points_to_allocate", points_to_allocate - 1)
                    ]
        
        # Bottom buttons
        hbox:
            xpos 600
            ypos 550
            spacing 50
            
            textbutton "Reset":
                style "cc_action_button"
                action [
                    SetDict(temp_stats, "agility", 0),
                    SetDict(temp_stats, "charisma", 0),
                    SetDict(temp_stats, "intellect", 0),
                    SetScreenVariable("points_to_allocate", 2)
                ]
            
            textbutton "Confirm":
                style "cc_action_button"
                sensitive points_to_allocate == 0
                action Return(temp_stats)

# Add styles for the level up screen
style level_up_auto_increment:
    font "gui/CreteRound-Regular.ttf"
    size 34
    color "#66AA66"  # Green color for automatic +1
    bold True
    xalign 0.0

style level_up_allocated:
    font "gui/CreteRound-Regular.ttf"
    size 34
    color "#6684a3"  # Blue color for allocated points
    bold True
    xalign 0.0

style level_up_formula:
    font "gui/CreteRound-Regular.ttf"
    size 34
    color "#AAAAAA"  # Grey color for formula elements (= and +)
    bold True
    xalign 0.0